package ellax.base.mvi

import android.content.Intent
import android.graphics.drawable.ColorDrawable
import android.os.Build
import android.os.Bundle
import android.view.*
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.viewbinding.ViewBinding
import com.blankj.utilcode.util.AppUtils
import ellax.base.IBaseView
import ellax.base.R
import java.lang.reflect.ParameterizedType

/**
 * Date: 2023/4/25 14:44
 * Author: funaihui@ellabook.cn
 * Describe: MVI架构的DialogFragment
 **/
open class BaseMVIDialogFragment<T : ViewBinding>  : DialogFragment(), IBaseView {

    protected lateinit var binding:T

    /**
     * 回调对象(可能是Activity、Fragment等)
     */
    private var callback: DialogCallback? = null

    /**
     * 请求码
     */
    private var requestCode = 0

    override fun onStart() {
        super.onStart()
        if (isFullScreenDialog()) {
            adaptNotchScreen()
        }
        dialog?.window?.let {
            setDialogWindowAttributes(it)
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setDialogStyle()
    }

    @Suppress("UNCHECKED_CAST")
    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View {
        val type = javaClass.genericSuperclass
        val clazz = (type as ParameterizedType).actualTypeArguments[0] as Class<T>
        val method = clazz.getMethod("inflate", LayoutInflater::class.java, ViewGroup::class.java, Boolean::class.java)
        binding = method.invoke(null, layoutInflater, container, false) as T
        return binding.root
    }

    /**
     * 显示对话框
     *
     * @param tag Fragment唯一标识
     */
    override fun show(manager: FragmentManager, tag: String?) {
        try {

            //在每个add事务前增加一个remove事务，防止连续的add
            manager.beginTransaction().remove(this).commit()
            // 解决Can not perform this action after onSaveInstanceState异常
            if (!manager.isStateSaved) {
                super.show(manager, tag)
            }
        } catch (exception: Exception) {
            //同一实例使用不同的tag会异常,这里捕获一下
            exception.printStackTrace()
        }
    }

    /**
     * 显示对话框
     */
    open fun show(manager: FragmentManager) {
        show(manager, getTagClassName())
    }

    /**
     * 是否是全屏弹框 (用于判断是否需要适配刘海屏)
     *
     * @return true: 是全屏弹框
     */
    protected open fun isFullScreenDialog(): Boolean {
        return false
    }

    final override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        // 业务view
        initView()
        // 初始化数据
        initData()
        // 初始化viewModel，处理view显示
        initViewModel()
        // 绑定事件监听
        initListener()
    }

    /**
     * 适配刘海屏
     */
    private fun adaptNotchScreen() {
        dialog?.window?.run {
            decorView.systemUiVisibility =
                View.SYSTEM_UI_FLAG_FULLSCREEN or View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                attributes.layoutInDisplayCutoutMode =
                    WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES
                attributes = attributes
            }
        }
    }

    override fun dismiss() {
        if (AppUtils.isAppForeground() || parentFragmentManager.isStateSaved) {
            super.dismissAllowingStateLoss()
        } else {
            super.dismiss()
        }
    }

    /**
     * 显示对话框
     * 通过该函数调用的对话框，关闭对话框请使用 [.dismissForResult]。
     *
     * @param manager     fragment 管理器
     * @param callback    回调
     * @param requestCode 请求code码
     */
    fun showForResult(
        manager: FragmentManager, requestCode: Int = 0, callback: DialogCallback?
    ) {
        this.show(manager)
        this.callback = callback
        this.requestCode = requestCode
    }


    protected fun dismissForResult(responseCode: Int) {
        this.dismissForResult(responseCode, null)
    }

    protected fun dismissForResult(responseCode: Int, intent: Intent?) {
        callback(responseCode, intent)
        dismiss()
    }


    /**
     * 关闭对话框，回掉结果
     *
     * @param responseCode 返回码
     * @param intent       返回的参数
     */
    private fun callback(responseCode: Int, intent: Intent?) {
        callback?.invoke(requestCode, responseCode, intent)
    }

    /**
     * 设置对话框样式
     */
    protected open fun setDialogStyle() {
        setStyle(STYLE_NO_FRAME, android.R.style.Theme_Dialog)
    }

    protected open fun setDialogWindowAttributes(window: Window?) {
        // 一定要设置Background，如果不设置，window属性设置无效
        window?.run {
            setBackgroundDrawable(ColorDrawable(0x00ffffff))
            attributes.windowAnimations = R.style.Animation_Scale
            val params = window.attributes
            setDialogLayoutParams(params)
            attributes = params

        }
    }


    /**
     * 初始化弹窗布局
     *
     * @param params 弹窗布局参数
     */
    protected open fun setDialogLayoutParams(params: WindowManager.LayoutParams) {
        // 子类实现
    }

    private fun getTagClassName(): String = javaClass.simpleName
}

typealias DialogCallback = (requestCode: Int, responseCode: Int, intent: Intent?) -> Unit