package ellax.base;

import android.app.Fragment;
import android.content.Intent;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import ellax.base.error.Error;
import ellax.base.error.ErrorTypes;
import ellax.base.helper.ToastHelper;
import ellax.base.lifecycle.Lifecycle;
import ellax.base.lifecycle.LifecycleListener;
import ellax.base.lifecycle.LifecycleManager;
import ellax.base.transition.Transition;
import ellax.base.transition.TransitionManager;
import ellax.base.transition.TransitionStateListener;

/**
 * created by dongdaqing 19-2-27 下午3:10
 */
public abstract class EllaBaseFrameworkFragment extends Fragment implements Lifecycle, IRouteDelegate, EllaView, TransitionStateListener {

    private BaseDelegate mDelegate;

    protected abstract int getLayoutFile();

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(getLayoutFile(), container, false);
        return beforeViewAdd(view);
    }

    @Override
    public void onDestroyView() {
        /**
         * @see EllaBaseFragment
         */
        mDelegate = null;
        super.onDestroyView();
    }

    public View beforeViewAdd(View view) {
        return view;
    }

    @Override
    public final void route(Intent intent, Bundle options, int requestCode) {
        getBaseDelegate().route(intent, options, requestCode);
    }

    @Override
    public final void finish(FinishOptions options) {
        getBaseDelegate().finish(options);
    }

    @Override
    public final void showProgress() {
        getBaseDelegate().showProgress();
    }

    @Override
    public final void update(long total, long current, float percent, String tag) {
        getBaseDelegate().update(total, current, percent, tag);
    }

    @Override
    public final void stopProgress() {
        getBaseDelegate().stopProgress();
    }

    protected IProgress provideProgress() {
        return null;
    }

    private BaseDelegate getBaseDelegate() {
        if (mDelegate == null)
            mDelegate = new BaseDelegate(this, provideProgress());
        return mDelegate;
    }

    @Override
    public void handleError(Error error) {
        if (error.isNetworkError()) {
            handleNetworkError(error);
        } else if (error.is(ErrorTypes.NO_DATA)) {
            handleEmptyDataError(error);
        } else {
            ToastHelper.showToast(error.getMessage());
        }
    }

    protected void handleNetworkError(Error error) {
        if (!TransitionManager.notifyPageStateChange(hashCode(), Transition.STATE_NETWORK_BROKEN)) {
            ToastHelper.showToast(error.getMessage());
        }
    }

    protected void handleEmptyDataError(Error error) {
        if (!TransitionManager.notifyPageStateChange(hashCode(), Transition.STATE_EMPTY)) {
            ToastHelper.showToast(error.getMessage());
        }
    }

    @Override
    public void onViewStateChanged(int state, View view) {

    }

    private LifecycleManager mLifecycleManager;

    public EllaBaseFrameworkFragment() {
        mLifecycleManager = new LifecycleManager();
    }

    @Override
    public void addListener(LifecycleListener listener) {
        mLifecycleManager.addListener(listener);
    }

    @Override
    public void removeListener(LifecycleListener listener) {
        mLifecycleManager.removeListener(listener);
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mLifecycleManager.onCreate();
    }

    @Override
    public void onStart() {
        super.onStart();
        mLifecycleManager.onLifecycleStart();
    }

    @Override
    public void onPause() {
        super.onPause();
        if (getActivity() != null && getActivity().isFinishing()) {
            mLifecycleManager.onLifecycleDestroy();
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        mLifecycleManager.onLifecycleStop();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        /**
         * 正常的Fragment的onDestroy，如果是Activity结束导致的调用，二次调用不会生效
         */
        mLifecycleManager.onLifecycleDestroy();
    }
}