package ellax.base;

import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import androidx.appcompat.app.AppCompatDialogFragment;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import ellax.base.error.Error;
import ellax.base.error.ErrorTypes;
import ellax.base.helper.ToastHelper;
import ellax.base.lifecycle.Lifecycle;
import ellax.base.lifecycle.LifecycleListener;
import ellax.base.lifecycle.LifecycleManager;
import ellax.base.transition.Transition;
import ellax.base.transition.TransitionManager;
import ellax.base.transition.TransitionStateListener;

import org.joor.Reflect;

/**
 * created by dongdaqing 19-5-10 下午2:05
 */
public class EllaBaseDialogFragment extends AppCompatDialogFragment implements Lifecycle, EllaView, TransitionStateListener {

    private BaseDelegate mDelegate;

    @Override
    public int show(FragmentTransaction transaction, String tag) {
        throw new RuntimeException("do not call this method,use #show(FragmentManager,String) instead");
    }

    @Override
    public void show(FragmentManager manager, String tag) {
        if (manager == null)
            return;

        if (isAdded()) {
            manager.beginTransaction().remove(this).commitAllowingStateLoss();
        }

        //这里的逻辑与DialogFragment保持一致
        Reflect.on(this).set("mDismissed", false);
        Reflect.on(this).set("mShownByMe", true);

        FragmentTransaction ft = manager.beginTransaction();
        ft.add(this, tag);
        ft.commitAllowingStateLoss();
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(getLayoutFile(), container, false);
        return beforeViewAdd(view, container);
    }

    protected View beforeViewAdd(View view, ViewGroup container) {
        //在view被添加到父容器之前做一些操作
        return view;
    }

    public int getLayoutFile() {
        return 0;
    }

    @Override
    public final void showProgress() {
        getBaseDelegate().showProgress();
    }

    @Override
    public final void update(long total, long current, float percent, String tag) {
        getBaseDelegate().update(total, current, percent, tag);
    }

    @Override
    public final void stopProgress() {
        getBaseDelegate().stopProgress();
    }

    protected IProgress provideProgress() {
        return null;
    }

    private BaseDelegate getBaseDelegate() {
        if (mDelegate == null)
            mDelegate = new BaseDelegate(this, provideProgress());
        return mDelegate;
    }

    @Override
    public void handleError(Error error) {
        if (error.isNetworkError()) {
            handleNetworkError(error);
        } else if (error.is(ErrorTypes.NO_DATA)) {
            handleEmptyDataError(error);
        } else {
            ToastHelper.showToast(error.getMessage());
        }
    }

    protected void handleNetworkError(Error error) {
        if (!TransitionManager.notifyPageStateChange(hashCode(), Transition.STATE_NETWORK_BROKEN)) {
            ToastHelper.showToast(error.getMessage());
        }
    }

    protected void handleEmptyDataError(Error error) {
        if (!TransitionManager.notifyPageStateChange(hashCode(), Transition.STATE_EMPTY)) {
            ToastHelper.showToast(error.getMessage());
        }
    }

    @Override
    public void onViewStateChanged(int state, View view) {

    }

    @Override
    public void onDestroyView() {
        /**
         * 避免在BaseDelegate中持有与当前View界面相关的内容，而导致内存泄漏或其他的错误持有数据的问题
         */
        mDelegate = null;
        super.onDestroyView();
    }

    private LifecycleManager mLifecycleManager;

    public EllaBaseDialogFragment() {
        mLifecycleManager = new LifecycleManager();
    }

    @Override
    public void addListener(LifecycleListener listener) {
        mLifecycleManager.addListener(listener);
    }

    @Override
    public void removeListener(LifecycleListener listener) {
        mLifecycleManager.removeListener(listener);
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mLifecycleManager.onCreate();
    }

    @Override
    public void onStart() {
        super.onStart();
        mLifecycleManager.onLifecycleStart();
    }

    @Override
    public void onPause() {
        super.onPause();
        if (getActivity() != null && getActivity().isFinishing()) {
            mLifecycleManager.onLifecycleDestroy();
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        mLifecycleManager.onLifecycleStop();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        /**
         * 正常的Fragment的onDestroy，如果是Activity结束导致的调用，二次调用不会生效
         */
        mLifecycleManager.onLifecycleDestroy();
    }
}
