package ellax.base;

import android.content.Intent;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;

import java.util.ArrayList;

import ellax.base.error.Error;
import ellax.base.error.ErrorTypes;
import ellax.base.helper.ToastHelper;
import ellax.base.lifecycle.Lifecycle;
import ellax.base.lifecycle.LifecycleListener;
import ellax.base.lifecycle.LifecycleManager;
import ellax.base.transition.Transition;
import ellax.base.transition.TransitionManager;
import ellax.base.transition.TransitionStateListener;
import ellax.notch.WindowHelperLegacyKt;

/**
 * created by dongdaqing 19-2-25 上午11:49
 * <p>
 * activity的基类
 */
public class EllaBaseActivity extends AppCompatActivity implements Lifecycle, EllaView, IRouteDelegate, TransitionStateListener, ActivityResultCallbackHolder {

    private static boolean enableFullScreen = true;
    private BaseDelegate mDelegate;
    private LifecycleManager mLifecycleManager;
    private ArrayList<ActivityResultCallback> mActivityResultCallbacks;

    public EllaBaseActivity() {
        mLifecycleManager = new LifecycleManager();
    }

    @Override
    public void setContentView(View view) {
        super.setContentView(view);
        makeWindowFullScreen();
    }

    @Override
    public void setContentView(View view, ViewGroup.LayoutParams params) {
        super.setContentView(view, params);
        makeWindowFullScreen();
    }

    @Override
    public void setContentView(int layoutResID) {
        super.setContentView(layoutResID);
        makeWindowFullScreen();
    }

    /**
     * 在全屏（不包含底部的导航栏）的状态下给这个View（通常是Toolbar）添加一个topPadding
     * 让statusBar能够不被我们的布局内容所遮挡
     */
    public int getStatusView() {
        return -1;
    }

    /**
     * 让当前界面全屏（不包含底部的导航栏）
     * <p>
     * 为什么要设置为全屏？
     * 现在为了最求美观在做设计时都会将状态栏和我们的界面设计成一个整体，这里提供一个方法来实现这样的需求
     */
    public void makeWindowFullScreen() {
        if (enableFullScreen)
            WindowHelperLegacyKt.makeWindowFullScreenL(this, darkStatusIcon(), getStatusView() == -1 ? null : findViewById(getStatusView()));
    }

    public static void setEnableFullScreen(boolean enableFullScreen) {
        EllaBaseActivity.enableFullScreen = enableFullScreen;
    }

    /**
     * 是否将状态栏图标设为黑色
     *
     * @return
     */
    public boolean darkStatusIcon() {
        return true;
    }

    @Override
    public final void showProgress() {
        getBaseDelegate().showProgress();
    }

    @Override
    public final void update(long total, long current, float percent, String tag) {
        getBaseDelegate().update(total, current, percent, tag);
    }

    @Override
    public final void stopProgress() {
        getBaseDelegate().stopProgress();
    }

    @Override
    public void handleError(Error error) {
        if (error.isNetworkError()) {
            handleNetworkError(error);
        } else if (error.is(ErrorTypes.NO_DATA)) {
            handleEmptyDataError(error);
        } else {
            ToastHelper.showToast(error.getMessage());
        }
    }

    protected void handleNetworkError(Error error) {
        if (!TransitionManager.notifyPageStateChange(hashCode(), Transition.STATE_NETWORK_BROKEN)) {
            ToastHelper.showToast(error.getMessage());
        }
    }

    protected void handleEmptyDataError(Error error) {
        if (!TransitionManager.notifyPageStateChange(hashCode(), Transition.STATE_EMPTY)) {
            ToastHelper.showToast(error.getMessage());
        }
    }

    protected IProgress provideProgress() {
        return null;
    }

    @Override
    public final void route(Intent intent, Bundle options, int requestCode) {
        getBaseDelegate().route(intent, options, requestCode);
    }

    @Override
    public final void finish(FinishOptions options) {
        getBaseDelegate().finish(options);
    }

    /**
     * 延迟初始化，避免progress中存在依赖于布局的view
     *
     * @return
     */
    private BaseDelegate getBaseDelegate() {
        if (mDelegate == null)
            mDelegate = new BaseDelegate(this, provideProgress());
        return mDelegate;
    }

    @Override
    public void onViewStateChanged(int state, View view) {
        //state change from Transition
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mLifecycleManager.onLifecycleStart();
    }

    @Override
    protected void onStop() {
        mLifecycleManager.onLifecycleStop();
        super.onStop();
    }

    @Override
    protected void onDestroy() {
        mLifecycleManager.onLifecycleDestroy();
        super.onDestroy();
    }

    @Override
    public void addListener(LifecycleListener listener) {
        mLifecycleManager.addListener(listener);
    }

    @Override
    public void removeListener(LifecycleListener listener) {
        mLifecycleManager.removeListener(listener);
    }

    @Override
    public void addActivityResultCallback(ActivityResultCallback callback) {
        if (mActivityResultCallbacks == null) {
            mActivityResultCallbacks = new ArrayList<>();
        }
        mActivityResultCallbacks.add(callback);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (mActivityResultCallbacks != null) {
            for (ActivityResultCallback callback : mActivityResultCallbacks) {
                callback.onActivityResult(requestCode, resultCode, data);
            }
        }
    }
}
