package com.ellabook.util.redis;

import com.alibaba.fastjson.JSONObject;
import com.ellabook.util.CodeUtil;
import org.apache.commons.beanutils.BeanUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.jedis.JedisConnection;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPoolConfig;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.TimeUnit;

public class RedisServiceUtil implements InitializingBean {
    private final static Map<Long, ThreadRedisTemplate> threadRedisTemplateMap = new HashMap<>();
    private final static StringRedisTemplate[] redisZoneTemplateArray = new StringRedisTemplate[16];
    private static Integer originalRedisZone = null;
    private static Long originalThreadId = null;
    @Autowired
    private StringRedisTemplate redisTemplate0;

    @Resource
    private RedisConnectionFactory connectionFactory;

    private Jedis jedis;

    @Override
    public void afterPropertiesSet() throws Exception {
        JedisConnectionFactory connectionFactory = (JedisConnectionFactory) redisTemplate0.getConnectionFactory();
        connectionFactory.setTimeout(3000);
        connectionFactory.setDatabase(RedisZoneEnum.ZONE0.getValue());
        JedisPoolConfig poolConfig = connectionFactory.getPoolConfig();
        poolConfig.setMaxIdle(500);
        poolConfig.setMinIdle(50);
        poolConfig.setMaxTotal(-1);
        poolConfig.setMaxWaitMillis(1000);
        redisZoneTemplateArray[0] = redisTemplate0;
        setDatabase(RedisZoneEnum.ZONE0);
        originalRedisZone = null;
    }

    private StringRedisTemplate createRedisZoneTemplate(RedisZoneEnum zoneEnum) throws Exception {
        StringRedisTemplate cloneRedisTemplate = (StringRedisTemplate) BeanUtils.cloneBean(redisTemplate0);
        cloneRedisTemplate.setConnectionFactory((JedisConnectionFactory) BeanUtils.cloneBean(redisTemplate0.getConnectionFactory()));
        cloneRedisTemplate.afterPropertiesSet();
        ((JedisConnectionFactory) cloneRedisTemplate.getConnectionFactory()).setDatabase(zoneEnum.getValue());
        return redisZoneTemplateArray[zoneEnum.getValue()] = cloneRedisTemplate;
    }

    public StringRedisTemplate getRedisTemplate() {
        long id = Thread.currentThread().getId();
        ThreadRedisTemplate threadRedisTemplate = threadRedisTemplateMap.get(id);
        if (threadRedisTemplate == null) {
            try {
                this.setDatabase(this.getDatabase());
                threadRedisTemplate = threadRedisTemplateMap.get(id);
            } catch (Exception e) {
            }
        }
        return threadRedisTemplate.getThreadRedisTemplate();
    }

    public StringRedisTemplate getRedisTemplate(RedisZoneEnum zoneEnum) throws Exception {
        int zone = zoneEnum == null ? -1 : zoneEnum.getValue();
        if (zone < 0 || zone > 15) {
            throw new Exception("error:zone=" + zone + ", redis zone value must between 0 and 15 !");
        }
        return redisZoneTemplateArray[zone] == null ? createRedisZoneTemplate(zoneEnum) : redisZoneTemplateArray[zone];
    }

    public int getDatabase() {
        ThreadRedisTemplate threadRedisTemplate = threadRedisTemplateMap.get(Thread.currentThread().getId());
        if (threadRedisTemplate == null) {
            try {
                setDatabase(originalRedisZone == null || originalRedisZone < 0 ? (originalRedisZone = 0) : originalRedisZone);
            } catch (Exception e) {
            }
        }
        StringRedisTemplate redisTemplate = threadRedisTemplateMap.get(Thread.currentThread().getId()).getThreadRedisTemplate();
        return ((JedisConnectionFactory) redisTemplate.getConnectionFactory()).getDatabase();
    }

    private void setDatabase(int zone) throws Exception {
        StringRedisTemplate redisTemplate = this.getRedisTemplate(RedisZoneEnum.getRedisZoneEnum(zone));
        long timeMillis = System.currentTimeMillis();
        Iterator<Map.Entry<Long, ThreadRedisTemplate>> iterator = threadRedisTemplateMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Long, ThreadRedisTemplate> next = iterator.next();
            Long key = next.getKey();
            if (key == null || (next.getValue().getExpire() / 1000) + 300 < timeMillis / 1000) {
                iterator.remove();
            }
        }
        ThreadRedisTemplate threadRedisTemplate = new ThreadRedisTemplate();
        threadRedisTemplate.setExpire(timeMillis);
        threadRedisTemplate.setThreadRedisTemplate(redisTemplate);
        threadRedisTemplate.setThread(Thread.currentThread());
        threadRedisTemplateMap.put(Thread.currentThread().getId(), threadRedisTemplate);
    }

    protected void setDatabase(RedisZoneEnum zoneEnum) throws Exception {
        long currentThreadId = Thread.currentThread().getId();
        if (originalThreadId == null) {
            originalThreadId = currentThreadId;
        }
        if (originalRedisZone == null && originalThreadId == currentThreadId) {
            originalRedisZone = zoneEnum.getValue();
        }
        setDatabase(zoneEnum == null ? -1 : zoneEnum.getValue());
    }

    @RequestMapping(value = "/set")
    public boolean set(final String key, final String value) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                connection.set(serializer.serialize(key), serializer.serialize(value));
                return true;
            }
        });
        return result;
    }

    @RequestMapping(value = "/setEx")
    public boolean setEx(final String key, final long expire, final String value) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                connection.setEx(serializer.serialize(key), expire, serializer.serialize(value));
                return true;
            }
        });
        return result;
    }

    @RequestMapping(value = "/setExpireNX")
    public boolean setExpireNX(final String key, final long expire, final String value) {
        boolean result = setNX(key, value);
        if (result) { //加锁成功，设置锁时间
            expire(key, expire);
        }
        return result;
    }

    @RequestMapping(value = "/setNX")
    public boolean setNX(final String key, final String value) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Boolean setNX = connection.setNX(serializer.serialize(key), serializer.serialize(value));
                return setNX;
            }
        });
        return result;
    }

    @RequestMapping(value = "/hSet")
    public boolean hSet(final String key, final String filed, final String value) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                connection.hSet(serializer.serialize(key), serializer.serialize(filed), serializer.serialize(value));
                return true;
            }
        });
        return result;
    }

    @RequestMapping(value = "/mSet")
    public boolean mSet(final String map) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                Map<String, String> map1 = JSONObject.parseObject(map, Map.class);
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Map<byte[], byte[]> emptyMap = Collections.emptyMap();
                map1.entrySet().iterator().forEachRemaining(m -> {
                    byte[] key = serializer.serialize(m.getKey());
                    byte[] value = serializer.serialize(m.getValue());
                    emptyMap.put(key, value);
                });
                connection.mSet(emptyMap);
                return true;
            }
        });
        return result;
    }


    @RequestMapping(value = "/get")
    public String get(final String key) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                byte[] value = connection.get(serializer.serialize(key));
                return serializer.deserialize(value);
            }
        });
        return result;
    }

    @RequestMapping(value = "/hGet")
    public String hGet(final String key, final String filed) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                byte[] value = connection.hGet(serializer.serialize(key), serializer.serialize(filed));
                return serializer.deserialize(value);
            }
        });
        return result;
    }

    @RequestMapping(value = "/mGet")
    public String mGet(final String list) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                List<String> list1 = JSONObject.parseObject(list, List.class);
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                byte[][] bytes = new byte[list1.size()][];
                for (int i = 0; list1.size() > 0 && i < list1.size(); i++) {
                    bytes[i] = list1.get(i).getBytes();
                }
                List<byte[]> list2 = connection.mGet(bytes);
                Map<String, String> map = new LinkedHashMap();
                int temp = 0;
                Iterator<byte[]> iterator = list2.iterator();
                while (iterator.hasNext()) {
                    byte[] next = iterator.next();
                    String a = next == null ? "" : serializer.deserialize(next);
                    map.put(list1.get(temp++), a);
                }
                return JSONObject.toJSONString(map);
            }
        });
        return result;
    }

    @RequestMapping(value = "/expire")
    public boolean expire(final String key, long expire) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        return redisTemplate.expire(key, expire, TimeUnit.SECONDS);
    }

    @RequestMapping(value = "/ttl")
    public long ttl(final String key) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Long value = connection.ttl(serializer.serialize(key));
                return serializer.deserialize(value.toString().getBytes());
            }
        });
        return Long.parseLong(result);
    }

    @RequestMapping(value = "/del")
    public long del(final String key) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Long value = connection.del(serializer.serialize(key));
                return serializer.deserialize(value.toString().getBytes());
            }
        });
        return Long.parseLong(result);
    }

    @RequestMapping(value = "/hDel")
    public long hDel(final String key, final String filed) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Long value = connection.hDel(serializer.serialize(key), serializer.serialize(filed));
                return serializer.deserialize(value.toString().getBytes());
            }
        });
        return Long.parseLong(result);
    }

    @RequestMapping(value = "/incr")
    public long incr(final String key) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Long value = connection.incr(serializer.serialize(key));
                return serializer.deserialize(value.toString().getBytes());
            }
        });
        return Long.parseLong(result);
    }

    @RequestMapping(value = "/incrByLong")
    public long incrByLong(final String key, final long num) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Long value = connection.incrBy(serializer.serialize(key), num);
                return serializer.deserialize(value.toString().getBytes());
            }
        });
        return Long.parseLong(result);
    }

    @RequestMapping(value = "/incrByDouble")
    public double incrByDouble(final String key, final double num) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                Double value = connection.incrBy(serializer.serialize(key), num);
                return serializer.deserialize(value.toString().getBytes());
            }
        });
        return Double.parseDouble(result);
    }

    @RequestMapping(value = "/getSet")
    public String getSet(final String key, final String filed) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        String result = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
                byte[] value = connection.getSet(serializer.serialize(key), serializer.serialize(filed));
                return serializer.deserialize(value);
            }
        });
        return result;
    }


    @RequestMapping(value = "/getMap")
    @ResponseBody
    public Map getMap(String key) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
//        System.out.println(key);
//        System.out.println(redisTemplate.opsForHash().entries(key));
        return redisTemplate.opsForHash().entries(key);
    }

    @RequestMapping(value = "/setMap")
    public void setMap(String key, @RequestBody Map map) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
//        System.out.println(map);
        redisTemplate.opsForHash().putAll(key, map);
    }

    /**
     * 获取当天唯一15位code编码，每次授权60
     *
     * @return
     */
    @RequestMapping(value = "/getCode")
    public synchronized String getCode() {
        String saveDate = get("REDIS:CODE:DATE");
        String saveIncr = get("REDIS:CODE:INCR");
        String saveIncrRecord = get("REDIS:CODE:INCR:RECORD");
        CodeUtil codeUtil = new CodeUtil(saveDate, saveIncr, saveIncrRecord).invoke();
        set("REDIS:CODE:DATE", codeUtil.getSaveDate());
        set("REDIS:CODE:INCR", Long.parseLong(codeUtil.getSaveIncr()) + 60 + "");
        set("REDIS:CODE:INCR:RECORD", codeUtil.getSaveIncrRecord());
        StringBuffer sb = new StringBuffer();
        codeUtil.setSaveIncr(get("REDIS:CODE:INCR"));
        int length = codeUtil.getSaveIncr().length();
        for (int i = 1; length < 6 && i < 6 - length; i++) {
            sb.append("0");
        }
        return (Long.parseLong(codeUtil.getSaveDate() + sb.toString() + get("REDIS:CODE:INCR")) - 59) + "";
    }


    @RequestMapping(value = "/jedisSet")
    public boolean jedisSet(String key, String requestId, long expireTime) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        boolean result = redisTemplate.execute((RedisCallback<Boolean>) connection -> tryGetDistributedLock(jedis(), key, requestId, expireTime));
        return result;
    }

    @RequestMapping(value = "/jedisDel")
    public boolean jedisDel(String key, String requestId) {
        StringRedisTemplate redisTemplate = getRedisTemplate();
        boolean result = redisTemplate.execute((RedisCallback<Boolean>) connection -> releaseDistributedLock(jedis(), key, requestId));
        return result;
    }

    public Set<String> scan(String pattern, Integer count) {
        Set<String> execute = redisTemplate0.execute((RedisCallback<Set<String>>) connection -> {
            Set<String> binaryKeys = new HashSet<>();
            Cursor<byte[]> cursor = connection.scan(new ScanOptions.ScanOptionsBuilder().match(pattern).count(count).build());
            while (cursor.hasNext()) {
                binaryKeys.add(new String(cursor.next()));
            }
            return binaryKeys;
        });
        return execute;
    }

    /**
     * 尝试获取分布式锁
     *
     * @param jedis      Redis客户端
     * @param lockKey    锁
     * @param requestId  请求标识
     * @param expireTime 超期时间
     * @return 是否获取成功
     */
    private boolean tryGetDistributedLock(Jedis jedis, String lockKey, String requestId, long expireTime) {
        String result = jedis.set(lockKey, requestId, "NX", "PX", expireTime);
        return "OK".equals(result);

    }

    /**
     * 释放分布式锁
     *
     * @param jedis     Redis客户端
     * @param lockKey   锁
     * @param requestId 请求标识
     * @return 是否释放成功
     */
    private boolean releaseDistributedLock(Jedis jedis, String lockKey, String requestId) {
        final Long releaseSuccess = 1L;
        String script = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";
        Object result = jedis.eval(script, Collections.singletonList(lockKey), Collections.singletonList(requestId));

        return releaseSuccess.equals(result);
    }

    /**
     * Jedis实例
     */
    private Jedis jedis() {
        if (jedis == null) {
            Field jedisField = ReflectionUtils.findField(JedisConnection.class, "jedis");
            ReflectionUtils.makeAccessible(jedisField);
            jedis = (Jedis) ReflectionUtils.getField(jedisField, connectionFactory.getConnection());
        }
        return jedis;
    }
}
