package com.ellabook.util;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 解析Excel
 *
 * @author LiBin
 **/
public class ReadExcelUtils {

    private static Logger logger = LoggerFactory.getLogger(ReadExcelUtils.class);
    private static Workbook wb;
    private static Sheet sheet;
    private static Row row;

    public static List<Map<Integer, Object>> fileToList(MultipartFile file) {
        try {
            String fileName = file.getOriginalFilename();
            String ext = fileName.substring(fileName.lastIndexOf("."));
            if (".xls".equals(ext)) {
                wb = new HSSFWorkbook(file.getInputStream());
            } else if (".xlsx".equals(ext)) {
                wb = new XSSFWorkbook(file.getInputStream());
            } else {
                wb = null;
            }
            return readExcelContent();
        } catch (Exception e) {
            logger.error("Exception", e);
            return null;
        }
    }

    public static boolean checkFile(MultipartFile file) {
        String fileName = file.getOriginalFilename();
        if (StringUtils.isBlank(fileName) || !StringUtils.contains(fileName, ".")) {
            return false;
        }
        String ext = fileName.substring(fileName.lastIndexOf("."));
        if (".xls".equals(ext) || ".xlsx".equals(ext)) {
            return true;
        }
        return false;
    }

    public static List<Map<Integer, Object>> inputStreamToList(String fileName, InputStream is) {
        String ext = fileName.substring(fileName.lastIndexOf("."));
        try {
            if (".xls".equals(ext)) {
                wb = new HSSFWorkbook(is);
            } else if (".xlsx".equals(ext)) {
                wb = new XSSFWorkbook(is);
            } else {
                wb = null;
            }
            return readExcelContent();
        } catch (Exception e) {
            logger.error("Exception", e);
            return null;
        }
    }

    private ReadExcelUtils(String filepath) {
        if (filepath == null) {
            return;
        }
        String ext = filepath.substring(filepath.lastIndexOf("."));
        try {
            InputStream is = new FileInputStream(filepath);
            if (".xls".equals(ext)) {
                wb = new HSSFWorkbook(is);
            } else if (".xlsx".equals(ext)) {
                wb = new XSSFWorkbook(is);
            } else {
                wb = null;
            }
        } catch (FileNotFoundException e) {
            logger.error("FileNotFoundException", e);
        } catch (IOException e) {
            logger.error("IOException", e);
        }
    }

    /**
     * 读取Excel表格表头的内容
     *
     * @return String 表头内容的数组
     */
    private String[] readExcelTitle() throws Exception {
        if (wb == null) {
            logger.info("Workbook对象为空");
            return null;
        }
        sheet = wb.getSheetAt(0);
        row = sheet.getRow(0);
        // 标题总列数
        int colNum = row.getPhysicalNumberOfCells();
        String[] title = new String[colNum];
        for (int i = 0; i < colNum; i++) {
            title[i] = row.getCell(i).getStringCellValue();
        }
        return title;
    }

    /**
     * 读取Excel数据内容
     *
     * @return Map 包含单元格数据内容的Map对象
     */
    private static List<Map<Integer, Object>> readExcelContent() throws Exception {
        if (wb == null) {
            logger.info("Workbook对象为空");
            return null;
        }
        List<Map<Integer, Object>> content = new ArrayList<>();

        sheet = wb.getSheetAt(0);
        // 得到总行数
        int rowNum = sheet.getLastRowNum();
        row = sheet.getRow(0);
        int colNum = row.getPhysicalNumberOfCells();
        // 正文内容应该从第二行开始,第一行为表头的标题
        for (int i = 1; i <= rowNum; i++) {
            row = sheet.getRow(i);
            if (null == row) {
                continue;
            }
            if (null == row.getCell(0) || StringUtils.isBlank(row.getCell(0).getStringCellValue())) {
                continue;
            }
            int j = 0;
            Map<Integer, Object> cellValue = new HashMap<>();
            while (j < colNum) {
                Object obj = getCellFormatValue(row.getCell(j));
                cellValue.put(j, obj);
                j++;
            }
            content.add(cellValue);
        }
        return content;
    }

    /**
     * 根据Cell类型设置数据
     *
     * @param cell 单元格
     * @return Object
     */
    private static Object getCellFormatValue(Cell cell) {
        Object cellValue = "";
        if (null == cell) {
            return cellValue;
        }
        // 判断当前Cell的Type
        switch (cell.getCellType()) {
            case Cell.CELL_TYPE_NUMERIC:
                cell.setCellType(Cell.CELL_TYPE_STRING);
                cellValue = cell.getStringCellValue();
                break;
            case Cell.CELL_TYPE_FORMULA: {
                if (org.apache.poi.ss.usermodel.DateUtil.isCellDateFormatted(cell)) {
                    // cellValue = cell.getDateCellValue().toLocaleString();
                    cellValue = cell.getDateCellValue();
                } else {
                    cellValue = String.valueOf(cell.getNumericCellValue());
                }
                break;
            }
            case Cell.CELL_TYPE_STRING:
                cellValue = cell.getRichStringCellValue().getString();
                break;
            default:
        }
        return cellValue;
    }

    /**
     * 导入Excel
     *
     * @return
     * @throws FileNotFoundException
     */
    public static String readExcel(InputStream in) throws FileNotFoundException {
        XSSFWorkbook xssfworkbook;
        JSONArray ja = new JSONArray();
        JSONObject jsonObject = null;
        Sheet sheet = null;
        Row row = null;
        try {
            xssfworkbook = new XSSFWorkbook(in);
            sheet = xssfworkbook.getSheetAt(0);

            int rowNum = sheet.getLastRowNum();
            row = sheet.getRow(0);
            int colNum = row.getPhysicalNumberOfCells();
            for (int i = 1; i <= rowNum; i++) {
                jsonObject = new JSONObject();
                row = sheet.getRow(i);
                int j = 0;
                while (j < colNum) {
                    jsonObject.put(String.valueOf(j), getCellFormatValue1(row.getCell(j)).trim());
                    j++;
                }
                ja.add(jsonObject);
            }

        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException ex) {
            ex.printStackTrace();
        }
        return ja.toString();
    }

    private static String getCellFormatValue1(Cell cell) {
        String cellValue = "";
        if (cell != null) {
            switch (cell.getCellType()) {
                case Cell.CELL_TYPE_NUMERIC: {
                    cellValue = String.valueOf(cell.getNumericCellValue());
                }
                case Cell.CELL_TYPE_FORMULA: {
                    if (HSSFDateUtil.isCellDateFormatted(cell)) {
                        // cellValue = cell.getDateCellValue().toLocaleString();
                        Date date = cell.getDateCellValue();
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
                        cellValue = sdf.format(date);
                    } /*
                     * else { cellValue =
                     * cell.getRichStringCellValue().getString();
                     *
                     * }
                     */
                    break;
                }
                case Cell.CELL_TYPE_STRING:
                    cellValue = cell.getRichStringCellValue().getString();
                    break;
                default:
                    cellValue = " ";
            }
        } else {
            cellValue = "";
        }
        return cellValue;
    }

    public static void main(String[] args) {
        try {
            String filepath = "C:\\Users\\ZC\\Desktop\\1.xlsx";
            InputStream in = new FileInputStream(filepath);
            String s = readExcel(in);
            System.out.println(s);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
    }
}
