package com.ellabook.template;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ellabook.entity.analysis.dto.KeyValDataDTO;
import com.ellabook.entity.analysis.vo.TimeSlotDataVO;
import com.mongodb.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.bson.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

public abstract class MongodbDaoBaseTemplate {
    public static String DATE_DAY_PATTERN = "yyyyMMdd";
    public static String EVENT_TIME_PATTERN = "yyyyMMddHHmmss";
    public static String EVENT_MILLIS_TIME_PATTERN = "yyyyMMddHHmmssSSS";
    protected String COLLECTION_NAME;
    protected String uidKey;
    protected String timeKey;
    @Autowired
    protected MongoTemplate mongoTemplate;

    public void setMongoTemplate(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    /**
     * 分组 的聚合函数列表
     */
    protected List<BasicDBObject> getGroupAggregationList(String startTime, String endTime, boolean containsEnd, boolean in, Collection<String> ids, BasicDBObject groupFields) {
        boolean noMatchDate = StringUtils.isBlank(startTime) && StringUtils.isBlank(endTime);

        BasicDBObject match = new BasicDBObject("$match", getTimeUidMatchObj(startTime, endTime, containsEnd, in, ids));
        BasicDBObject group = new BasicDBObject("$group", groupFields);

        return noMatchDate ? Arrays.asList(group) : Arrays.asList(match, group);
    }

    /**
     * 时间条件，uid条件 的查询对象
     */
    protected BasicDBObject getTimeUidMatchObj(String startTime, String endTime, boolean containsEnd, boolean in, Collection<String> ids) {
        BasicDBObject matchObj = timeMatchObj(startTime, endTime, containsEnd);
        matchObj.put(uidKey, uidMatchObj(in, ids));
        return matchObj;
    }

    protected BasicDBObject timeMatchObj(String startTime, String endTime, boolean containsEnd) {
        startTime = getDay(startTime);
        endTime = getDay(endTime);

        return new BasicDBObject(timeKey,
                StringUtils.equalsIgnoreCase(startTime, endTime)
                        ? objForTimeEq(startTime)
                        : new BasicDBObject("$gte", startTime).append(getLteOrLt(containsEnd), endTime));
    }

    protected String getLteOrLt(boolean contains) {
        return contains ? "$lte" : "$lt";
    }

    //mongo.in:3w+ ok
    protected BasicDBObject uidMatchObj(boolean in, Collection<String> ids) {
        return in ? new BasicDBObject("$in", ids) : new BasicDBObject("$nin", CollectionUtils.isEmpty(ids) ? Arrays.asList(null, "") : ids);
    }

    /**
     * mongo 不支持 yyyy-MM-dd 的 范围内查询
     */
    protected String getDay(String day) {
        return StringUtils.replace(day, "-", "");
    }

    protected Object objForTimeEq(String startTime) {
        return startTime;
    }

    /**
     * 列表只1条，获取（只有一个分组的）dataKey数据
     */
    protected JSONArray getData(BasicDBList dbList, String dataKey) {
        for (Object result : dbList) {
            if (result == null) {
                continue;
            }
            try {
                JSONObject jsonObject = JSONObject.parseObject(result.toString());
                return jsonObject.getJSONArray(dataKey);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return new JSONArray();
    }

    /**
     * 获取 转化类型后 的列表
     */
    protected <T> List<T> getListData(List dbList, Class<T> clazz) {
        List<T> rs = new ArrayList<>();
        for (Object result : dbList) {
            if (result == null) {
                continue;
            }
            try {
                T t = JSONObject.parseObject(result.toString(), clazz);
                rs.add(t);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return rs;
    }

    /**
     * 获取 分组id 的列表
     */
    protected List getListDataForGroupId(List dbList) {
        List<KeyValDataDTO> listData = getListData(dbList, KeyValDataDTO.class);
        List list = new ArrayList();
        for (KeyValDataDTO item : listData) {
            list.add(item.getId());
        }
        return list;
    }

    /**
     * 聚合计算
     */
    protected BasicDBList aggregation(List<BasicDBObject> list, String collectionName) {
        try {
            Cursor dbCursor = getCursor(list, collectionName);
            BasicDBList dbList = new BasicDBList();
            while (dbCursor.hasNext()) {
                DBObject dbObject = dbCursor.next();
                dbList.add(dbObject);
            }
            return dbList;
        } catch (Exception e) {
            e.printStackTrace();
            return new BasicDBList();
        }
    }

    /**
     * 根据keyName 将各分组内的值 统一返回
     */
    protected Set aggregationByKey(List<BasicDBObject> list, String keyName) {
        try {
            Cursor dbCursor = getCursor(list, COLLECTION_NAME);//groupId=null:>16m
            Set keyList = new HashSet();
            while (dbCursor.hasNext()) {
                DBObject dbObject = dbCursor.next();
                Object keyVal = dbObject.get(keyName);
                keyList.add(keyVal);//groupId="_id":50w+ slow
            }
            return keyList;
        } catch (Exception e) {
            e.printStackTrace();
            return new HashSet();
        }
    }

    /**
     * 根据keyName 将各分组内的集合 合并返回
     */
    protected Set<String> aggregationByKeyForColl(List<BasicDBObject> list, String keyName) {
        try {
            Cursor dbCursor = getCursor(list, COLLECTION_NAME);
            Set keyList = new HashSet();
            while (dbCursor.hasNext()) {
                DBObject dbObject = dbCursor.next();
                BasicDBList arr = (BasicDBList) dbObject.get(keyName);
                keyList.addAll(arr);
            }
            return keyList;
        } catch (Exception e) {
            e.printStackTrace();
            return new HashSet();
        }
    }

    protected Set<String> aggregationForGroupId(List<BasicDBObject> list) {
        return aggregationByKey(list, "_id");
    }

    /**
     * 获取游标
     */
    private Cursor getCursor(List<BasicDBObject> list, String collectionName) {
        AggregationOptions build = AggregationOptions.builder().allowDiskUse(true).batchSize(5000)
                .outputMode(AggregationOptions.OutputMode.CURSOR)
                .build();
        return mongoTemplate.getCollection(collectionName).aggregate(list, build);
    }

    protected Query getQueryByParamMap(Map<String, Object> paramMap) {
        Query query = new Query();
        if (paramMap != null) {

            Criteria criteria = new Criteria();
            for (Map.Entry<String, Object> kv : paramMap.entrySet()) {
                String key = kv.getKey();
                Object value = kv.getValue();
                if (key != null && value != null) {
                    criteria.and(key).is(kv.getValue());
                }
            }

            query.addCriteria(criteria);
        }
        return query;
    }

    protected Document getCmd(String mapFunction, String reduceFunction, Document match) {
        Document command = new Document();
        command.put("mapreduce", COLLECTION_NAME);
        command.put("query", match);
        command.put("map", mapFunction);
        command.put("reduce", reduceFunction);
        command.put("out", new Document("inline", 1));

        return command;
    }

    /**
     * MR 计算 时间分组下的活跃uid数
     */
    protected List<TimeSlotDataVO> countTimeGroupActiveUser(boolean isDay, Document match) {
        String mapFunction = "function() {emit({uid:this.uid,time:this." + timeKey + ".substring" + getGroupCodeVal(isDay) + "}, null); } ";
        String reduceFunction = "function (key, values) { return null; }";

        Document document = runCommand(getCmd(mapFunction, reduceFunction, match));
        List<Document> results = document.get("results", List.class);
        if (CollectionUtils.isNotEmpty(results)) {
            Map<String, Integer> map = results.stream().collect(Collectors.groupingBy(o -> {
                Document id = (Document) o.get("_id");
                String time = id.getString("time");
                return time;
            }, Collectors.summingInt(o -> 1)));
            List<TimeSlotDataVO> list = new ArrayList<>();
            for (Map.Entry<String, Integer> kv : map.entrySet()) {
                list.add(new TimeSlotDataVO(kv.getKey(), new BigDecimal(kv.getValue())));
            }
            return list;
        }

        return new ArrayList<>();
    }

    protected Document runCommand(Document cmd) {
        return null;
    }

    private String getGroupCodeVal(boolean isDay) {
        if (isDay) {
            return "(8,10).concat(':00')";//hour
        }
        return "(0,8)";
    }
}
