package com.ellabook.template;

import com.constants.BookRedisConstantUtil;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.zip.CRC32;
import java.util.zip.Checksum;

import static com.ellabook.util.ListUtil.group;

public abstract class ADSBaseTemplate<T> {
    public final static String DATE_TIME_PATTERN = "yyyyMMdd HH:mm:ss";
    // 缓存表最大记录数
    public final static int ANALYTIC_CACHE_MAX_SIZE = 300000;
    //缓存表日期范围数
    public final static int ANALYTIC_CACHE_DAY_NUM = 7;
    //重试次数
    public final static int RETRY_COUNT = 3;
    // 一级分区个数
    public int ANALYTIC_PARTITION_SIZE;
    // batch insert 条数
    protected int ANALYTIC_BATCH_INSERT_SIZE;
    protected Logger logger;

    /**
     * 提前进行optimize table
     */
    public void optimize() {
        Map<String, String> optimizeMap = optimizeInADS();
        if (optimizeMap != null) {
            String msyText = optimizeMap.get("Msg_text");
            if (!StringUtils.startsWithIgnoreCase(msyText, "OK")) {
                logger.error("optimizeInADS {}", msyText);
            }
        }
    }

    /**
     * 在 ADS optimize 进行的操作
     */
    protected abstract Map<String, String> optimizeInADS();

    /**
     * 批量新增
     */
    public boolean insertBatchConcurrent(Map<Integer, Set<T>> partitionGroupMap, int checkNum) {
        boolean success = true;
        if (partitionGroupMap.isEmpty()) {
            logger.info("userBehaviorAnalysis uidGroup empty");
            return true;
        }

        //尽量保证一批记录的目标hash分区相同
        for (Map.Entry<Integer, Set<T>> kv : partitionGroupMap.entrySet()) {
            Set<T> set = kv.getValue();
            int size = set.size();
            if (size > checkNum) {
                List<T> list;
                final ReentrantLock lock = new ReentrantLock();
                try {
                    lock.lock();
                    list = new ArrayList<>(set);
                    set.clear();
                } finally {
                    lock.unlock();
                }

                sortAndGroupAndInsert(list);
            }
        }
        return success;
    }

    /**
     * 批量新增
     */
    public boolean insertBatch(Map<Integer, List<T>> partitionGroupMap) {
        boolean success = true;
        if (partitionGroupMap.isEmpty()) {
            logger.info("userBehaviorAnalysis uidGroup empty");
            return true;
        }

        //尽量保证一批记录的目标hash分区相同
        for (Map.Entry<Integer, List<T>> kv : partitionGroupMap.entrySet()) {
            List<T> list = kv.getValue();
            sortAndGroupAndInsert(list);
        }
        return success;
    }

    /**
     * 主键排序，分组，批量新增
     */
    protected boolean sortAndGroupAndInsert(List<T> list) {
        boolean success = true;
        sortByPrimaryKey(list);
        List<List<T>> groupList = group(list, ANALYTIC_BATCH_INSERT_SIZE);
        for (List<T> gList : groupList) {
            int listSize = gList.size();

            retry:
            for (int i = 0; i < RETRY_COUNT; i++) {
                try {
                    int insertBatch = insertBatchInADS(gList);
                    if (listSize != insertBatch) {
                        logger.error("insertBatchInADS {}:{}", listSize, insertBatch);
                        success = false;
                    }

                    break retry;
                } catch (Exception e) {
                    logger.error("insertBatchInADS {}", e);
                }
            }
        }
        return success;
    }

    /**
     * 在 ADS 批量新增 进行的操作
     */
    protected abstract int insertBatchInADS(List<T> list);

    /**
     * 添加到分区集合
     */
    public boolean insertToCollConcurrent(T t, Map<Integer, Set<T>> partitionGroupMap) {
        if (encodeBASE64(t)) {
            return false;
        }

        int partitionNum = getPartitionNum(partitionHashKey(t));
        Set<T> set = partitionGroupMap.get(partitionNum);

        if (set == null) {
            set = ConcurrentHashMap.newKeySet();//safe

            final ReentrantLock reentrantLock = new ReentrantLock();
            try {
                reentrantLock.lock();
                Set<T> existSet = partitionGroupMap.get(partitionNum);
                if (existSet == null) {
                    partitionGroupMap.put(partitionNum, set);
                } else {
                    set = existSet;
                }
            } finally {
                reentrantLock.unlock();
            }

        }

        set.add(t);
        return true;
    }

    /**
     * 添加到分区集合
     */
    public boolean insertToColl(T t, Map<Integer, List<T>> partitionGroupMap) {
        if (!encodeBASE64(t)) {
            return false;
        }

        int partitionNum = getPartitionNum(partitionHashKey(t));
        List<T> list = partitionGroupMap.get(partitionNum);
        if (list == null) {
            list = new ArrayList<>();
            partitionGroupMap.put(partitionNum, list);
        }
        list.add(t);
        return true;
    }

    /**
     * BASE64 编码 进行的操作
     */
    protected boolean encodeBASE64(T t) {
        //TODO
        return true;
    }

    /**
     * 主键排序
     */
    protected void sortByPrimaryKey(List<T> list) {
        //插入时主键是严格递增或近似递增的，也可以提升实时写入速度。
        //TODO
    }

    /******************************* 分区列相关 ****************************************/
    /**
     * 1级分区hashKey
     */
    protected abstract Object partitionHashKey(T t);

    /**
     * @param partitionColVal
     * @return calculte partition num basing on partition column value
     */
    protected int getPartitionNum(Object partitionColVal) {
        long crc32 = (partitionColVal == null ? getCRC32("-1") : getCRC32(partitionColVal.toString()));
        return (int) (crc32 % ANALYTIC_PARTITION_SIZE);
    }

    protected static final long getCRC32(String value) {
        Checksum checksum = new CRC32();
        byte[] bytes = value.getBytes();
        checksum.update(bytes, 0, bytes.length);
        return checksum.getValue();
    }

    /**
     * 2级分区key
     */
    protected int getSubKey() {
        return 0;
    }

    /******************************* 缓存表相关 ****************************************/
    /**
     * cacheHead
     */
    public static String getCacheHead(String cacheId) {
        return StringUtils.isNotBlank(cacheId) ? "/*+cache_id=" + cacheId + "*/" : null;
    }

    /**
     * 创建缓存表
     */
    public void createCache() {
        Integer countCacheTable = countForCreateCacheTableInADS(ANALYTIC_CACHE_DAY_NUM);
        if (countCacheTable < ANALYTIC_CACHE_MAX_SIZE) {
            Date now = new Date();
            Date startDate = DateUtils.addDays(now, -ANALYTIC_CACHE_DAY_NUM);
            String startDay = DateFormatUtils.format(startDate, "yyyy-MM-dd");

            String redisKey = BookRedisConstantUtil.REDIS_ANALYSIS_ANALYTIC_CACHE;
            String cacheId = redisGetCacheId(redisKey);
            for (int i = 0; i < RETRY_COUNT; i++) {
                try {
                    String cacheTableId = createCacheTableInADS(startDay, getCacheHead(cacheId));

                    redisSetCacheId(redisKey, cacheTableId);
                    return;
                } catch (Exception e) {
                    redisDelCacheId(redisKey);

                    e.printStackTrace();
                    logger.error("createCache {}", e);
                }
            }
        }
    }

    protected void redisDelCacheId(String redisKey) {
        //
    }

    protected void redisSetCacheId(String redisKey, String cacheTableId) {
        //
    }

    protected String redisGetCacheId(String redisKey) {
        return null;
    }

    /**
     * 在ADS 创建缓存表 进行的操作
     */
    protected String createCacheTableInADS(String startDay, String cacheHead) {
        return null;
    }

    /**
     * 在ADS 统计缓存表记录数 进行的操作
     */
    protected Integer countForCreateCacheTableInADS(int cacheDayNum) {
        return 0;
    }
}