package com.ellabook.util;

import com.ellabook.constraint.inf.SetDefault;
import com.ellabook.constraint.inf.SetDefaultContainer;
import org.apache.commons.collections.CollectionUtils;
import org.hibernate.validator.HibernateValidator;
import org.hibernate.validator.internal.engine.path.PathImpl;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import java.lang.annotation.Annotation;
import java.util.*;
import java.util.stream.Collectors;

public class ValidatorUtil {
    private Validator validator;
    private boolean failFast = true;

    private ValidatorUtil() {
    }

    /**
     * 获取验证处理器
     *
     * @return
     */
    public ValidatorUtil getValidator(boolean failFast) {
        this.failFast = failFast;
        validator = failFast ? validator : Validation.byProvider(HibernateValidator.class)
                .configure()
                .failFast(failFast)
                .buildValidatorFactory()
                .getValidator();
        return this;
    }

    public static ValidatorUtil getInstance() {
        ValidatorUtil validatorUtil = new ValidatorUtil();
        validatorUtil.validator = Validation.byProvider(HibernateValidator.class)
                .configure()
                .failFast(true)
                .buildValidatorFactory()
                .getValidator();
        return validatorUtil;
    }

    /**
     * 参数校验结果已String形式返回
     *
     * @param obj
     * @param groups
     * @param <T>
     * @return
     * @throws Exception
     */
    public <T> String validAndGetMessage(T obj, Class<?>... groups) throws Exception {
        isSetDefaultValidatorMust(obj);
        Set<ConstraintViolation<T>> validate = validator.validate(obj, groups);
        return getValidatorMessageStr(validate);
    }

    public <T> String validAndGetMessage(T obj,
                                         String propertyName,
                                         Class<?>... groups) throws Exception {
        isSetDefaultValidatorMust(obj);
        Set<ConstraintViolation<T>> validate = validator.validateProperty(obj, propertyName, groups);
        return getValidatorMessageStr(validate);
    }

    public <T> String validAndGetMessage(Class<T> beanType,
                                         String propertyName,
                                         Object value,
                                         Class<?>... groups) throws Exception {
        Set<ConstraintViolation<T>> validate = validator.validateValue(beanType, propertyName, value, groups);
        return getValidatorMessageStr(validate);
    }

    /**
     * 参数校验结果已list形式返回
     *
     * @param obj
     * @param groups
     * @param <T>
     * @return
     * @throws Exception
     */
    public <T> List<String> validAndGetMessageList(T obj, Class<?>... groups) throws Exception {
        isSetDefaultValidatorMust(obj);
        Set<ConstraintViolation<T>> validate = validator.validate(obj, groups);
        return getValidatorMessageList(validate);
    }

    public <T> List<String> validAndGetMessageList(T obj,
                                                   String propertyName,
                                                   Class<?>... groups) throws Exception {
        isSetDefaultValidatorMust(obj);
        Set<ConstraintViolation<T>> validate = validator.validateProperty(obj, propertyName, groups);
        return getValidatorMessageList(validate);
    }

    public <T> List<String> validAndGetMessageList(Class<T> beanType,
                                                   String propertyName,
                                                   Object value,
                                                   Class<?>... groups) throws Exception {
        Set<ConstraintViolation<T>> validate = validator.validateValue(beanType, propertyName, value, groups);
        return getValidatorMessageList(validate);
    }

    /**
     * 参数校验结果已map形式返回
     *
     * @param obj
     * @param groups
     * @param <T>
     * @return
     * @throws Exception
     */
    public <T> Map<String, String> validAndGetMessageMap(T obj, Class<?>... groups) throws Exception {
        isSetDefaultValidatorMust(obj);
        Set<ConstraintViolation<T>> validate = validator.validate(obj, groups);
        return getValidatorMessageMap(validate);
    }

    public <T> Map<String, String> validAndGetMessageMap(T obj,
                                                         String propertyName,
                                                         Class<?>... groups) throws Exception {
        isSetDefaultValidatorMust(obj);
        Set<ConstraintViolation<T>> validate = validator.validateProperty(obj, propertyName, groups);
        return getValidatorMessageMap(validate);
    }

    public <T> Map<String, String> validAndGetMessageMap(Class<T> beanType,
                                                         String propertyName,
                                                         Object value,
                                                         Class<?>... groups) throws Exception {
        Set<ConstraintViolation<T>> validate = validator.validateValue(beanType, propertyName, value, groups);
        return getValidatorMessageMap(validate);
    }

    /**
     * @param obj
     * @param <T>
     */
    private <T> void isSetDefaultValidatorMust(T obj) {
        List<Annotation> annotations = new ArrayList<>(Arrays.asList());
        if (!CollectionUtils.isEmpty(annotations)) {
            Iterator<Annotation> iterator = annotations.iterator();
            while (iterator.hasNext()) {
                String name = iterator.next().getClass().getName();
                if (SetDefaultContainer.getInstance().get(name) != null && failFast == true) {
                    getValidator(false);
                    break;
                }
            }
        }
    }

    /**
     * 将验证结果封装字符串
     *
     * @param constraintViolationList
     * @return
     */
    public static <T> String getValidatorMessageStr(Set<ConstraintViolation<T>> constraintViolationList) throws Exception {
        if (CollectionUtils.isEmpty(constraintViolationList)) {
            return null;
        }
        constraintViolationList = setFieldDefaultValue(constraintViolationList);
        String messages = constraintViolationList.stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(";\\n"));
        return messages;
    }


    /**
     * 将验证结果封装为list
     *
     * @param constraintViolationList
     * @return
     */
    public static <T> List<String> getValidatorMessageList(Set<ConstraintViolation<T>> constraintViolationList) throws Exception {
        if (CollectionUtils.isEmpty(constraintViolationList)) {
            return null;
        }
        constraintViolationList = setFieldDefaultValue(constraintViolationList);
        List<String> messages = constraintViolationList.stream().map(ConstraintViolation::getMessage).collect(Collectors.toList());
        return messages;
    }

    /**
     * 将验证结果封装为map
     *
     * @param constraintViolationList
     * @return
     */
    public static <T> Map<String, String> getValidatorMessageMap(Set<ConstraintViolation<T>> constraintViolationList) throws Exception {
        if (CollectionUtils.isEmpty(constraintViolationList)) {
            return null;
        }
        constraintViolationList = setFieldDefaultValue(constraintViolationList);
        Map<String, String> map = new HashMap<>();
        constraintViolationList.iterator().forEachRemaining(tConstraintViolation -> {
            PathImpl propertyPath = (PathImpl) tConstraintViolation.getPropertyPath();
            String message = tConstraintViolation.getMessage();
            map.put(propertyPath.toString(), message);
        });
        return map;
    }

    private static <T> Set<ConstraintViolation<T>> setFieldDefaultValue(Set<ConstraintViolation<T>> constraintViolationList) throws Exception {
        Iterator<ConstraintViolation<T>> iterator = constraintViolationList.iterator();
        while (iterator.hasNext()) {
            ConstraintViolation<T> tConstraintViolation = iterator.next();
            Annotation annotation = tConstraintViolation.getConstraintDescriptor().getAnnotation();
            String name = annotation.annotationType().getName();
            SetDefault setDefault = SetDefaultContainer.getInstance().get(name);
//            Map<String, SetDefault> requests = SetDefaultContainer.requests;
            if (setDefault != null) {
                setDefault.setDefault(tConstraintViolation, annotation);
                iterator.remove();
            }
        }
        return constraintViolationList;
    }

    public static void main(String[] args) {
        try {
            Person person = new Person();
            String s = ValidatorUtil.getInstance().getValidator(false).validAndGetMessage(person);
            System.out.println(person);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


}
