package com.ellabook.util;

import net.coobird.thumbnailator.Thumbnails;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.poi.util.IOUtils;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

public class ImgTools {

    /**
     * 将图片压缩到指定大小以内
     *
     * @param srcImgData 源图片数据
     * @param maxSize    目的图片大小
     * @return 压缩后的图片数据
     */
    public static byte[] compressUnderSize(byte[] srcImgData, long maxSize, long maxWidth, long maxHeight) {
        byte[] imgData = Arrays.copyOf(srcImgData, srcImgData.length);
        try {
            byte[] compress = compress(imgData, maxSize, maxWidth, maxHeight);
            return compress;
        } catch (IOException e) {
            throw new IllegalStateException("压缩图片过程中出错，请及时联系管理员！", e);
        }

    }

    /**
     * 按照 宽高 比例压缩，
     *
     * @param srcImgData 待压缩图片输入流
     * @param maxSize    压缩后图片大小的最大值
     * @param maxWidth
     * @param maxHeight
     * @return
     * @throws IOException
     */
    public static byte[] compress(byte[] srcImgData, long maxSize, long maxWidth, long maxHeight) throws IOException {
        BufferedImage bi = ImageIO.read(new ByteArrayInputStream(srcImgData));
        int width = (bi.getWidth()); // 源图宽度
        int height = (bi.getHeight()); // 源图高度
        //压缩比例0～1
        double scale = Double.min(maxSize * 1.0 / srcImgData.length, Double.min(maxWidth * 1.0 / width, maxHeight * 1.0 / height));
        scale = getScale(maxWidth, maxHeight, width, height, scale, 1.0);//压缩比列的精确值为0.01
        width = (int) (bi.getWidth() * scale);
        height = (int) (bi.getHeight() * scale);
        if (0 < scale && scale < 1 && width > 0 && height > 0) {
            Image image = bi.getScaledInstance(width, height, Image.SCALE_SMOOTH);
            BufferedImage tag = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);

            Graphics g = tag.getGraphics();
            g.setColor(Color.RED);
            g.drawImage(image, 0, 0, null); // 绘制处理后的图
            g.dispose();

            ByteArrayOutputStream bOut = new ByteArrayOutputStream();
            ImageIO.write(tag, "JPEG", bOut);
            srcImgData = bOut.toByteArray();
            if (srcImgData.length > maxSize || width > maxWidth || height > maxHeight) {
                srcImgData = compress(srcImgData, maxSize, maxWidth, maxHeight);
            }
        }
        return srcImgData;
    }


    /**
     * 压缩比列算法
     *
     * @param maxWidth
     * @param maxHeight
     * @param width
     * @param height
     * @param scale
     * @param prefixScale
     * @return
     */
    private static double getScale(long maxWidth, long maxHeight, int width, int height, double scale, double prefixScale) {
        int w = (int) (width * scale);
        int h = (int) (height * scale);
        if (scale < 1 && scale > 0 && w < maxWidth && h < maxHeight) {//计算满足条件的最大压缩刻度值
            prefixScale = scale;
            scale = 1.1 * scale;//1.1为比例的加速度值
            scale = getScale(maxWidth, maxHeight, width, height, scale, prefixScale);
        } else {
            scale = prefixScale;
        }
        return scale;
    }

    /**
     * 等比例压缩指定路径下所有图片
     *
     * @param path
     * @throws Exception
     */
    public static void compressImg(String path, long maxSize, long maxWidth, long maxHeight) throws Exception {
        List<File> imgFileList = FileUtil.getImgFileList(path);
        if (!CollectionUtils.isEmpty(imgFileList)) {
            Iterator<File> iterator = imgFileList.iterator();
            while (iterator.hasNext()) {
                File next = iterator.next();
                byte[] bytes = IOUtils.toByteArray(new FileInputStream(next));
                byte[] data = ImgTools.compressUnderSize(bytes, maxSize, maxWidth, maxHeight);
                FileUtils.writeByteArrayToFile(next, data);
            }
        }
    }

    /**
     * 等比例压缩指定路径下所有图片
     *
     * @param path
     * @throws Exception
     */
    public static void compressImgByThumbnails(String path, long maxSize, long maxWidth, long maxHeight) throws Exception {
        List<File> imgFileList = FileUtil.getImgFileList(path);
        if (!CollectionUtils.isEmpty(imgFileList)) {
            Iterator<File> iterator = imgFileList.iterator();
            while (iterator.hasNext()) {
                File next = iterator.next();
                System.out.println("---" + next.getAbsolutePath());
                byte[] bytes1 = IOUtils.toByteArray(new FileInputStream(next));
                byte[] bytes = Arrays.copyOf(bytes1, bytes1.length);

                BufferedImage bi = ImageIO.read(new ByteArrayInputStream(bytes));
                int width = (bi.getWidth()); // 源图宽度
                int height = (bi.getHeight()); // 源图高度
                if (bytes.length <= maxSize && width <= maxWidth && height <= maxHeight) {
                    continue;
                }
                double scale = Double.min(maxWidth * 1.0d / width, maxHeight * 1.0d / height);
                scale = scale > 1 ? 1 : scale;
                thumbnails(next, bytes, maxSize, maxWidth, maxHeight, scale);
            }
        }
    }

    private static void thumbnails(File next, byte[] bytes, long maxSize, long maxWidth, long maxHeight, double scale) throws IOException {
        boolean qualityCompress = false;
        boolean sizeCompress = false;
        boolean sizeIncrease = false;

        if (scale < 1) {//标记为固定比率图片质量压缩
            sizeCompress = true;
        }

        if (bytes.length > maxSize) {
            qualityCompress = true;
        }

        System.out.println("源文件初始大小：" + next.getName() + ": " + BigDecimalUtil.round(next.length() / (1024 * 1024.0), 4) + "M");
        Thumbnails.of(next).scale(scale).outputQuality(1).toFile(next);//第一次压缩
        System.out.println("第一次压缩大小：" + next.getName() + ": " + BigDecimalUtil.round(next.length() / (1024 * 1024.0), 4) + "M");

        byte[] bytes1 = IOUtils.toByteArray(new FileInputStream(next));
        if (bytes1.length <= maxSize) {//第一次压缩已经符合条件需求
            return;
        }

        if (bytes.length <= bytes1.length) {
            byte[] data = ImgTools.compressUnderSize(bytes, maxSize, maxWidth, maxHeight);
            FileUtils.writeByteArrayToFile(next, data);
            return;
        }

        if (qualityCompress) {
            int flag = bytes.length > maxSize ? 1 : (bytes.length < maxSize ? -1 : 0);
            compress(next, maxSize, scale, 0.0, 1.0, flag, sizeIncrease, sizeCompress);
        }
    }

    private static void compress(File next, long maxSize, double scale, double start, double end, int flag, boolean sizeIncrease, boolean sizeCompress) throws IOException {
        double outputQuality = 1.0d;
        if (sizeCompress) {
            outputQuality = (outputQuality - 0.001) * 0.99;
        } else {
            outputQuality = (start + end) / 2;
        }
        if (flag == 1) {
            end = (start + end) / 2;
        } else if (flag == -1) {
            start = (start + end) / 2;
        } else {
            return;
        }
        if (sizeIncrease) {
            scale = scale * 0.99;
            outputQuality = 1;
        }
        Thumbnails.of(next).scale(scale).outputQuality(outputQuality).toFile(next);
        byte[] bytes1 = IOUtils.toByteArray(new FileInputStream(next));
        long length = bytes1.length;
        System.out.println("第n次压缩--------scale:" + BigDecimalUtil.round(scale, 4) + "---length:"
                + BigDecimalUtil.round(length / (1024 * 1024.0), 4) + "M" + "---outputQuality:" + outputQuality);
        if (scale < 1 && length < maxSize) {
            return;
        }
        if (flag == 1 && length > maxSize) {
            compress(next, maxSize, scale, start, end, 1, sizeIncrease, sizeCompress);
        }
        if (flag == 1 && length < maxSize) {
            compress(next, maxSize, scale, start, end, -1, sizeIncrease, sizeCompress);
        }
        if (flag == -1 && length > maxSize) {
            Thumbnails.of(next).scale(scale).outputQuality(outputQuality).toFile(next);
            compress(next, maxSize, scale, start, end, 0, sizeIncrease, sizeCompress);
        }
        if (flag == -1 && length < maxSize) {
            compress(next, maxSize, scale, start, end, -1, sizeIncrease, sizeCompress);
        }
    }

    /**
     * 等比例压缩指定路径下所有图片
     *
     * @param path
     * @throws Exception
     */
    public static void compressImgKeepTranslucent(String path, long maxWidth, long maxHeight) throws Exception {
        List<File> imgFileList = FileUtil.getImgFileList(path);
        if (!CollectionUtils.isEmpty(imgFileList)) {
            Iterator<File> iterator = imgFileList.iterator();
            while (iterator.hasNext()) {
                File next = iterator.next();
                compressKeepTranslucent(next, next, maxWidth, maxHeight);
            }
        }
    }

    public static void compressKeepTranslucent(File fromFile, File toFile, long outputMaxWidth, long outputMaxHeight) {
        try {
            BufferedImage sourceBufferImage = ImageIO.read(fromFile);

            /**
             * 等比例缩小图片（输出图片宽度必须小于outputMaxWidth，输出图片高度必须小于outputMaxHeight）
             */
            int width = sourceBufferImage.getWidth(null);
            int height = sourceBufferImage.getHeight(null);
            double widthRate = outputMaxWidth * 1.0 / width;
            double heightRate = outputMaxHeight * 1.0 / height;
            double rate = widthRate < heightRate ? widthRate : heightRate;
            if (rate < 1) {
                width = (int) (((double) sourceBufferImage.getWidth(null)) * rate);
                height = (int) (((double) sourceBufferImage.getHeight(null)) * rate);
            }

            Image image = sourceBufferImage.getScaledInstance(width, height, Image.SCALE_SMOOTH);
            /**
             * 重置图片大小（如：4096*4096）
             */
            BufferedImage reSizeBufferedImage = new BufferedImage(width, height, BufferedImage.TYPE_4BYTE_ABGR);
            Graphics2D g2d = reSizeBufferedImage.createGraphics();
            reSizeBufferedImage = g2d.getDeviceConfiguration().createCompatibleImage(width, height, BufferedImage.TRANSLUCENT);
            g2d.dispose();
            g2d = reSizeBufferedImage.createGraphics();
            g2d.drawImage(image, 0, 0, null);
            g2d.dispose();
            /**
             * 创建TYPE_USHORT_555_RGB格式压缩图，透明像素点会被转换为黑色
             *
             * <p>555格式其实是16位位图中的一种。16位位图最多有65536种颜色。每个色素用16位（2个字节）表示。这种格式叫作高彩色
             * ，或叫增强型16位色，或64K色。16位中，最低的5位表示蓝色分量，中间的5位表示绿色分量，高的5位表示红色分量，一共
             * 占用了15位，最高的一位保留，设为0。在555格式下，红、绿、蓝的掩码分别是：0x7C00、0x03E0、0x001F（在BufferedImage
             * 源码中也有定义）。</p>
             */
            BufferedImage compressedBufferedImage = new BufferedImage(width, height, BufferedImage.TYPE_USHORT_555_RGB);
            g2d = compressedBufferedImage.createGraphics();
            g2d.drawImage(reSizeBufferedImage, 0, 0, null);

            /**
             *将TYPE_USHORT_555_RGB格式图片对应的透明像素点还原
             */
            int resultPixel[] = new int[width * height];
            int reSizePixel[] = new int[width * height];
            int compressedPixel[] = new int[width * height];
            reSizePixel = reSizeBufferedImage.getRGB(0, 0, width, height, reSizePixel, 0, width);
            compressedPixel = compressedBufferedImage.getRGB(0, 0, width, height, compressedPixel, 0, width);
            for (int i = 0; i < compressedPixel.length; i++) {
                resultPixel[i] = reSizePixel[i] == 0 ? reSizePixel[i] : compressedPixel[i];
            }
            /**
             * 绘制还原透明像素点后的压缩图
             */
            BufferedImage resultBufferedImage = new BufferedImage(width, height, BufferedImage.TYPE_4BYTE_ABGR);
            resultBufferedImage.setRGB(0, 0, width, height, resultPixel, 0, width);
            g2d = resultBufferedImage.createGraphics();
            g2d.drawImage(resultBufferedImage, 0, 0, null);
            g2d.dispose();
            /**
             * 输出最终的压缩图
             */
            ImageIO.write(resultBufferedImage, "png", toFile);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}