package com.ellabook.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;


public class HttpConnectUtil {
    private static Logger log = LoggerFactory.getLogger(HttpConnectUtil.class);

    /**
     * 请求URL链接接口，参数格式为XML格式
     *
     * @param url    接口URL
     * @param map    参数map
     * @param method 方法 POST/GET
     * @return 返回xml解析后的map
     */
    public static Map<String, String> getConnectionXmlResponse(String url
            , TreeMap<String, String> map, String method) {
        Map<String, String> result = null;
        String xml = mapToXml(map);

        HttpURLConnection connection;
        try {
            connection = (HttpURLConnection) new URL(url).openConnection();
            connection.setRequestMethod(method);
            connection.setDoInput(true);
            connection.setDoOutput(true);
            connection.setUseCaches(false);
            connection.setRequestProperty("content-type", "application/x-www-form-urlencoded");

            if (map.size() != 0) {
                OutputStream os = connection.getOutputStream();
                os.write(xml.getBytes("utf-8"));
                os.close();
            } else {
                connection.connect();
            }

            InputStream is = connection.getInputStream();
            String xmlResult = inputStreamToString(is);
            result = xmlToMap(xmlResult);
        } catch (MalformedURLException e) {
            e.printStackTrace();
        } catch (ProtocolException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return result;
    }


    /**
     * 参数传入body请求
     *
     * @param url
     * @param body
     * @param method
     * @return
     */
    public static String getConnectionFromBody(String url, Object body, String method) {
        String result = "";

        HttpURLConnection connection;
        try {
            connection = (HttpURLConnection) new URL(url).openConnection();
            connection.setRequestMethod(method);
            connection.setDoInput(true);
            connection.setDoOutput(true);
            connection.setUseCaches(false);
            connection.setRequestProperty("content-type", "application/x-www-form-urlencoded");

            if (null != body && body.toString().length() != 0) {
                OutputStream os = connection.getOutputStream();
                os.write(body.toString().getBytes("utf-8"));
                os.close();
            } else {
                connection.connect();
            }

            InputStream is = connection.getInputStream();
            result = inputStreamToString(is);

        } catch (MalformedURLException e) {
            e.printStackTrace();
        } catch (ProtocolException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return result;
    }

    /**
     * 从输入流按UTF-8编码读取字符
     *
     * @param is 输入流
     * @return 返回读取字符串
     */
    public static String inputStreamToString(InputStream is) {
        StringBuffer sb = new StringBuffer();

        try {
            InputStreamReader reader = new InputStreamReader(is, "utf-8");
            BufferedReader br = new BufferedReader(reader);

            String str = "";
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
            log.debug("输入流解析结果:" + sb);

            br.close();
            reader.close();
            is.close();
            is = null;
        } catch (UnsupportedEncodingException e) {
            log.debug("输入流编码格式转换错误");
            e.printStackTrace();
        } catch (IOException e) {
            log.debug("字符流读取数据错误");
            e.printStackTrace();
        }
        return sb.toString();
    }

    /**
     * 通过url建立连接，以json格式字符串返回结果
     *
     * @param url
     * @param paramMap
     * @param method
     * @return
     */
    public static String getConnectionJsonResponse(String url, Map<String, Object> paramMap, String method) {
        JSONObject json = null;

        String connectURL = createURLWithParam(url, paramMap);

        try {
            URL getUrl = new URL(connectURL);
            HttpURLConnection conn = (HttpURLConnection) getUrl
                    .openConnection();
            conn.setRequestMethod(method);
            conn.setRequestProperty("Content-Type",
                    "application/x-www-form-urlencoded");
            conn.setDoInput(true);
            conn.setDoOutput(true);
            conn.connect();

            InputStream is = conn.getInputStream();
            int size = is.available();
            byte[] b = new byte[size];
            is.read(b);

            String message = new String(b, "utf-8");
            json = (JSONObject) JSON.parse(message);

            log.debug("return json:" + json.toString());
        } catch (MalformedURLException e) {
            log.error("获取openid失败 : " + e.getMessage());
        } catch (IOException e) {
            log.error("getOpenId连接打开失败: " + e.getMessage());
        }
        return json.toString();
    }

    /**
     * 根据URL和参数MAP,生成带参URL
     *
     * @param url
     * @param paramMap
     * @return
     * @author:lihe
     * @date:2017-4-27
     */
    public static String createURLWithParam(String url, Map<String, Object> paramMap) {

        StringBuffer sb = new StringBuffer(url).append("?");

        for (Entry<String, Object> entry : paramMap.entrySet()) {
            sb.append(entry.getKey()).append("=")
                    .append(entry.getValue()).append("&");
        }
        sb.deleteCharAt(sb.length() - 1);

        return sb.toString();
    }

    /**
     * map转换成xml格式String
     *
     * @param map
     * @return
     */
    public static String mapToXml(Map<String, String> map) {
        StringBuffer xml = new StringBuffer("<xml>");
        for (Entry<String, String> entry : map.entrySet()) {
            xml.append("<").append(entry.getKey()).append(">")
                    .append(entry.getValue())
                    .append("</").append(entry.getKey()).append(">");
        }
        xml.append("</xml>");
        return xml.toString();
    }

    /**
     * XML转化为Map
     *
     * @param xml
     * @return
     */
    public static Map<String, String> xmlToMap(String xml) {
        xml = xml.replaceFirst("encoding=\".*\"", "encoding=\"UTF-8\"");
        if (StringUtils.isEmpty(xml)) {
            return null;
        }
        Map<String, String> map = null;

        try {

            InputStream in = new ByteArrayInputStream(xml.getBytes("utf-8"));
            SAXReader reader = new SAXReader();
            Document doc = reader.read(in);
            Element root = doc.getRootElement();
            map = parseXML(root, new HashMap<String, String>());

        } catch (DocumentException e) {
            e.printStackTrace();
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }

        return map;
    }

    /**
     * 将xml解析成map键值对
     *
     * @param ele 需要解析的xml对象
     * @param map 入参为空，用于内部迭代循环使用
     * @return
     */
    private static Map<String, String> parseXML(Element ele,
                                                Map<String, String> map) {

        for (Iterator<?> i = ele.elementIterator(); i.hasNext(); ) {
            Element node = (Element) i.next();

            if (node.attributes() != null && node.attributes().size() > 0) {
                for (Iterator<?> j = node.attributeIterator(); j.hasNext(); ) {
                    Attribute item = (Attribute) j.next();

                    map.put(item.getName(), item.getValue());
                }
            }
            if (node.getText().length() > 0) {
                map.put(node.getName(), node.getText());
            }
            if (node.elementIterator().hasNext()) {
                parseXML(node, map);
            }
        }
        return map;
    }

    /**
     * 微信签名生成
     *
     * @param map 所有参数的TreeMap
     * @param key 商户密钥
     * @return
     */
    public static String createWechatSign(TreeMap<String, String> map,
                                          String key) {
        StringBuffer signBuffer = new StringBuffer();

        for (Entry<String, String> entry : map.entrySet()) {
            signBuffer.append(entry.getKey()).append("=")
                    .append(entry.getValue()).append("&");
        }
        signBuffer.append("key=").append(key);
        return DigestUtils.md5Hex(signBuffer.toString()).toUpperCase();
    }
}
