package com.ellabook.util;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.junit.Test;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.*;

public class DateUtil {

    public static String DEFAULT_FORMAT = "yyyy-MM-dd HH:mm:ss";
    public static String DEFAULT_DATE_FORMAT = "yyyy-MM-dd";
    public static String DEFAULT_BEGIN_TIME = " 00:00:00";
    public static String DEFAULT_END_TIME = " 23:59:59";

    /**
     * 时间戳转日期
     *
     * @param timeStamp
     * @return
     */
    public static Date timeStamp2Date(String timeStamp) {
        if (timeStamp == null || timeStamp.isEmpty() || timeStamp.equals("null")) {
            return new Date();
        }
        return new Date(Long.valueOf(timeStamp));
    }

    /**
     * 日期转换成字符串
     */
    public static String dateToString(Date date, String format) {
        SimpleDateFormat df = new SimpleDateFormat(format);
        return df.format(date);
    }

    /**
     * 日期转换成字符串
     */
    public static String dateToString(Date date) {
        return dateToString(date, DEFAULT_FORMAT);
    }

    /**
     * 字符串转换成日期
     */
    public static Date stringToDate(String date, String format) {
        SimpleDateFormat df = new SimpleDateFormat(format);
        try {
            return df.parse(date);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 字符串转换成日期
     */
    public static Date stringToDate(String date) {
        return stringToDate(date, DEFAULT_FORMAT);
    }

    public static Integer leftTime(Date date) {
        Long beginTimes = dateConvert(new Date(), DEFAULT_BEGIN_TIME);
        Long endTimes = dateConvert(date, DEFAULT_END_TIME);
        Long leftTime = (endTimes - beginTimes) / 1000 / 60 / 60; //剩余小时数

        if (leftTime % 24 == 0) {
            leftTime = leftTime / 24;
        } else {
            leftTime = leftTime / 24 + 1;
        }
        return Integer.valueOf(String.valueOf(leftTime));
    }

    /**
     * 日期处理
     *
     * @return
     */
    public static Long dateConvert(Date date, String endTime) {
        String dateStr = dateToString(date, DEFAULT_DATE_FORMAT) + endTime;
        Date nowDate = stringToDate(dateStr);
        return nowDate.getTime();
    }

    /**
     * 当天最后一秒
     *
     * @return
     */
    public static Date todayLastTime() {
        DateFormat sdf = new SimpleDateFormat(DEFAULT_FORMAT);
        LocalDate today = LocalDate.now();
        Date dat = new Date();

        String dateStr = today + DEFAULT_END_TIME;
        Date nowDate = null;
        try {
            nowDate = sdf.parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return nowDate;
    }


    /**
     * 计算系统日期与目标日期的相隔天数
     *
     * @param tar 与系统时间对比的目标日期
     * @return 相隔天数, 参数无效返回-1
     * author:
     */
    public static int getIntervalDay(Date tar) {
        int ret = -1;
        Calendar calNow = Calendar.getInstance();
        if (null != tar
                && tar.before(calNow.getTime())) {//参数有效

            //获得指定时间的Calendar
            Calendar calTar = Calendar.getInstance();
            calTar.setTime(tar);

            long millisNow = calNow.getTimeInMillis();
            long millisTar = tar.getTime();

            //指定时间小于系统时间才处理， 否则返回-1
            if (millisTar < millisNow) {
                //86400000 = 24*60*60*1000;
                ret = (int) ((millisNow - millisTar) / (86400000));
            }
        }
        return ret;
    }

    public static String getLastMonth() {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, -1);

        return format.format(calendar.getTime());
    }

    public static String getLastMonth(String month) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");
        Calendar calendar = Calendar.getInstance();
        try {
            calendar.setTime(format.parse(month));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        calendar.add(Calendar.MONTH, -1);

        return format.format(calendar.getTime());
    }

    public static String getLastDay() {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_YEAR, -1);
        return format.format(calendar.getTime());
    }

    public static String getTodayDateStr() {
        SimpleDateFormat format = new SimpleDateFormat(DEFAULT_DATE_FORMAT);
        return format.format(new Date());
    }

    public static String getNextDay(String day, int n) {
        SimpleDateFormat sdf = new SimpleDateFormat(DEFAULT_DATE_FORMAT);

        Calendar calendar = Calendar.getInstance();//获取日历实例
        try {
            calendar.setTime(sdf.parse(day));
            calendar.add(Calendar.DAY_OF_MONTH, n);  //设置为后n天
        } catch (ParseException e) {
            e.printStackTrace();
        }
        String tomorrow = sdf.format(calendar.getTime());//获得后n天
        return tomorrow;
    }

    /**
     * 获取本周的第一天和最后一天
     *
     * @param data 时间
     * @return 结果
     */
    public static Date[] getFirstAndLastOfWeek(Date data) {
        Date[] dates = new Date[2];
        Calendar cal = Calendar.getInstance();
        cal.setTime(data);
        int d = 0;
        if (cal.get(Calendar.DAY_OF_WEEK) == 1) {
            d = -6;
        } else {
            d = 2 - cal.get(Calendar.DAY_OF_WEEK);
        }
        cal.add(Calendar.DAY_OF_WEEK, d);
        // 所在周开始日期
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd 00:00:00");
        SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd 23:59:59");
        SimpleDateFormat sdf2 = new SimpleDateFormat(DEFAULT_FORMAT);
        String dateS = sdf.format(cal.getTime());
        try {
            dates[0] = sdf.parse(dateS);
            cal.add(Calendar.DAY_OF_WEEK, 6);
            dateS = sdf1.format(cal.getTime());
            // 所在周结束日期
            dates[1] = sdf2.parse(dateS);
        } catch (ParseException e) {
        }
        return dates;

    }

    /**
     * 获取某月份的最后一天
     *
     * @param date 日期
     * @return 最后一天
     */
    public static Date getLastdayOfMonth(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd 23:59:59");
        SimpleDateFormat sdf1 = new SimpleDateFormat(DEFAULT_FORMAT);
        Calendar ca = Calendar.getInstance();
        try {
            ca.setTime(date);
            ca.set(Calendar.DAY_OF_MONTH, 1);
            ca.add(Calendar.MONTH, 1);
            ca.add(Calendar.DAY_OF_MONTH, -1);
            String dateS = sdf.format(ca.getTime());
            return sdf1.parse(dateS);
        } catch (ParseException e) {
            return ca.getTime();
        }
    }

    /**
     * 获取某月份的第一天
     *
     * @param date 日期
     * @return 第一天
     */
    public static Date getFirstdayOfMonth(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd 00:00:00");
        Calendar ca = Calendar.getInstance();
        ca.setTime(date);
        ca.set(Calendar.DAY_OF_MONTH, 1);
        try {
            String dateS = sdf.format(ca.getTime());
            return sdf.parse(dateS);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return ca.getTime();
    }

    /**
     * 返回两个日期相差的天数
     *
     * @param before 时间1
     * @param after  时间2
     * @return 相差的天数（正数）
     */
    public static long daysBetweenNew(Date before, Date after) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(before);
        long time1 = cal.getTimeInMillis();
        cal.setTime(after);
        long time2 = cal.getTimeInMillis();
        long between_days = (time2 - time1) / (1000 * 3600 * 24);
        return Math.abs(between_days);
    }

    public static boolean isSameDay(Date date1, Date date2) {
        String dateString1 = dateToString(date1, "yyyy-MM-dd");
        String dateString2 = dateToString(date2, "yyyy-MM-dd");
        if (StringUtils.equalsIgnoreCase(dateString1, dateString2)) {
            return true;
        }
        return false;
    }


    public static Date plusDay(int num, Date newDate) {
        try {
            Calendar ca = Calendar.getInstance();
            ca.setTime(newDate);
            ca.add(Calendar.DATE, num);// num为增加的天数，可以改变的
            newDate = ca.getTime();
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return null;
        }
        return newDate;
    }

    /**
     * 获取字符串年
     * 例："2018"
     *
     * @return
     */
    public static String getYearString() {
        Date date = new Date();
        SimpleDateFormat yyyy = new SimpleDateFormat("yyyy");
        String format = yyyy.format(date);
        return format;
    }


    private static boolean isThisTime(Date date, String pattern) {
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        String param = sdf.format(date);//参数时间
        String now = sdf.format(new Date());//当前时间
        if (param.equals(now)) {
            return true;
        }
        return false;
    }

    /**
     * 判断
     * 本月返回本月,其他月返回  本月  1月 2月 ....
     * 其他返回年带月 2017年12月
     *
     * @param date
     * @return
     */
    public static String dateToMonth(Date date) {
        boolean thisTime = isThisTime(date, "yyyy-MM");
        //本月
        if (thisTime) {
            return "本月";
        } else {
            boolean yyyy = isThisTime(date, "yyyy");
            if (yyyy) {//本年
                String s = getDataMonth(date);
                return s + "月";
            }
            String dataYear = getDataYear(date);
            String dataMonth = getDataMonth(date);
            return dataYear + "年" + dataMonth + "月";
        }

    }

    public static String dateFormat(Date date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return format.format(date);
    }

    public static String dateFormatStr(Date date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMddHHmmssSSS");
        return format.format(date);
    }

    public static Date getDate(String date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm");
        try {
            return format.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    public static Date perseDate(Date date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        try {
            return format.parse(format.format(date));
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    public static String addCalendarStr(String startTimeStr, Integer feild, Integer amount) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Date endTime = null;

        try {
            Date startTime = format.parse(startTimeStr);
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(startTime);
            calendar.add(feild, amount);
            endTime = calendar.getTime();
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return format.format(endTime);
    }

    public static String formatConvert(String dateStr, String formatStr, String format2Str) {
        SimpleDateFormat format = new SimpleDateFormat(formatStr);
        SimpleDateFormat format2 = new SimpleDateFormat(format2Str);
        Date date = null;
        try {
            date = format.parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return format2.format(date);
    }

    public static Integer getDaysOfDatesStr(String startDate, String endDate, String formatStr) {
        SimpleDateFormat format = new SimpleDateFormat(formatStr);
        Integer days = 0;

        try {
            Date start = format.parse(startDate);
            Date end = format.parse(endDate);
            days = (int) ((end.getTime() - start.getTime()) / (1000 * 60 * 60 * 24));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return days;
    }

    public static Date getNextYearToday() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.YEAR, 1);
        return calendar.getTime();
    }


    public static String getNextDay(String date, Integer subDay) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        try {
            Calendar c = Calendar.getInstance();
            c.setTime(format.parse(date));
            c.add(Calendar.DAY_OF_MONTH, subDay);// 今天+1天
            Date tomorrow = c.getTime();
//            System.out.println("明天是:" + format.format(tomorrow));
            return format.format(tomorrow);
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 中英文星期转换
     *
     * @param i
     * @return
     */
    public static int getCNDayOfWeek(int i) {
        HashMap<Integer, Integer> map = new HashMap<>();
        map.put(1, 7);
        map.put(2, 1);
        map.put(3, 2);
        map.put(4, 3);
        map.put(5, 4);
        map.put(6, 5);
        map.put(7, 6);
        return map.get(i);
    }

    public static String getDataYear(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.YEAR) + "";
    }

    public static String getDataMonth(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return (c.get(Calendar.MONTH) + 1) + "";
    }

    public static String getDataDay(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.DAY_OF_MONTH) + "";
    }

    public static String getDateHour(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.get(Calendar.HOUR_OF_DAY) + "";
    }

    public static long getLongOfEarlierDate(Date start, Date end) {
        if (start.getTime() > end.getTime()) {
            return end.getTime();
        }
        return start.getTime();
    }

    public static long getLongOfLaterDate(Date start, Date end) {
        if (start.getTime() < end.getTime()) {
            return end.getTime();
        }
        return start.getTime();
    }


    /**
     * data 转换
     *
     * @param datetime
     * @return 2018-05-09 周三
     */
    public static String dateToWeek(Date datetime) {
        SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd");
        java.util.Date date = new java.util.Date();
        String str = f.format(datetime);
        String[] weekDays = {"周日", "周一", "周二", "周三", "周四", "周五", "周六"};
        // 获得一个日历
        Calendar cal = Calendar.getInstance();
        Date datet = null;
        try {
            datet = f.parse(str);
            cal.setTime(datet);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        //// 指示一个星期中的某天。
        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w < 0) {
            w = 0;
        }
        return str + " " + weekDays[w];
    }

    public static String weekOfDay(Date datetime) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(datetime);
        String[] weekDays = {"日", "一", "二", "三", "四", "五", "六"};
        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w < 0) {
            w = 0;
        }
        return weekDays[w];
    }

    /**
     * 将目标日期的时间设置为：23:59:59
     *
     * @param date
     * @return
     */
    public static Date getDayEndTime(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DATE), 23, 59, 59);
        return calendar.getTime();
    }

    /**
     * 将目标日期的时间设置为：00:00:00
     *
     * @param date
     * @return
     */
    public static Date getDayStartTime(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DATE), 0, 0, 0);
        return calendar.getTime();
    }

    /**
     * 将目标日期的时间设置为：00:00:00.000
     *
     * @param date
     * @return
     */
    public static Date getDayStartTimeForMilliSecond(Date date) {
        GregorianCalendar gregorianCalendar = new GregorianCalendar(Locale.CHINA);
        gregorianCalendar.setTime(date);
        gregorianCalendar.set(gregorianCalendar.get(Calendar.YEAR), gregorianCalendar.get(Calendar.MONTH), gregorianCalendar.get(Calendar.DATE), 0, 0, 0);
        gregorianCalendar.set(Calendar.MILLISECOND, 0);
        return gregorianCalendar.getTime();
    }

    /**
     * 获取指定时间的前一天零点，和当天零点
     *
     * @param date 指定时间
     * @return List 角标0为前一天0点时间数据， 角标1为当天0点时间数据
     */
    public static List<Date> getPreAndCurrentZeroOClock(Date date) {
        List<Date> dateResult = Lists.newArrayList(null, null);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        dateResult.set(1, calendar.getTime());
        calendar.add(Calendar.DAY_OF_YEAR, -1);
        dateResult.set(0, calendar.getTime());
        return dateResult;
    }

    /**
     * 判断当前时间是否在[startTime, endTime]区间，注意时间格式要一致
     *
     * @param nowTime   当前时间
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return
     * @author jqlin
     */
    public static boolean isEffectiveDate(Date nowTime, Date startTime, Date endTime) {
        if (nowTime.getTime() == startTime.getTime()
                || nowTime.getTime() == endTime.getTime()) {
            return true;
        }

        Calendar date = Calendar.getInstance();
        date.setTime(nowTime);

        Calendar begin = Calendar.getInstance();
        begin.setTime(startTime);

        Calendar end = Calendar.getInstance();
        end.setTime(endTime);

        if (date.after(begin) && date.before(end)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 判断当前时间(不含日期)是否在[strDateBegin, strDateEnd]区间
     *
     * @param time         当前时间
     * @param strDateBegin 开始时间
     * @param strDateEnd   结束时间
     * @return
     * @author jqlin
     */
    public static boolean isInDate(long time, String strDateBegin, String strDateEnd) {
        Calendar calendar = Calendar.getInstance();
        // 处理开始时间
        String[] startTime = strDateBegin.split(":");
        calendar.set(Calendar.HOUR_OF_DAY, Integer.valueOf(startTime[0]));
        calendar.set(Calendar.MINUTE, Integer.valueOf(startTime[1]));
        calendar.set(Calendar.SECOND, Integer.valueOf(startTime[2]));
        calendar.set(Calendar.MILLISECOND, 0);
        long startTimeL = calendar.getTimeInMillis();
        // 处理结束时间
        String[] endTime = strDateEnd.split(":");
        calendar.set(Calendar.HOUR_OF_DAY, Integer.valueOf(endTime[0]));
        calendar.set(Calendar.MINUTE, Integer.valueOf(endTime[1]));
        calendar.set(Calendar.SECOND, Integer.valueOf(endTime[2]));
        calendar.set(Calendar.MILLISECOND, 0);
        long endTimeL = calendar.getTimeInMillis();
        return time >= startTimeL && time <= endTimeL;
    }

    public static void main(String[] args) {
//        Calendar cal = Calendar.getInstance();
//        cal.add(Calendar.DATE, 1);
//        System.out.println("----------" + perseDate(cal.getTime()));
        Date now = new Date();
        Date todayBegin = getDayStartTime(new Date());
        Date testD = stringToDate("2018-08-17 09:45:49", "yyyy-MM-dd HH:mm:ss");

        System.out.println("todayBegin==========" + dateFormat(todayBegin));
        System.out.println("effectiveDate===========" + isEffectiveDate(testD, todayBegin, now));
        System.out.println("now==================" + dateFormat(now));

    }

    /**
     * 获取日期的毫秒数
     *
     * @param date
     * @return
     */
    public static long getTimeByDate(Date date) {
        date = date != null ? date : new Date();

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
//        calendar.add(calendar.DAY_OF_MONTH, 1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime().getTime();
    }

    /**
     * 获取下周第一天日期
     */
    public static Date getNextWeekFirstDay() {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        calendar.setTime(calendar.getTime());
        calendar.add(Calendar.DAY_OF_WEEK, 7);
        return calendar.getTime();
    }

    /**
     * 获取下月第一天日期
     */
    public static Date getNextMonthFirstDay() {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.MONTH, 1);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        return calendar.getTime();
    }

    /**
     * 获取昨天前日期
     */
    public static Date getNextDay() {
        Calendar calendar = Calendar.getInstance();
        int day = calendar.get(Calendar.DATE);
        calendar.set(Calendar.DATE, day + 1);
        return calendar.getTime();
    }


    /**
     * 获取昨天前日期
     */
    public static String getNextDayToStr() {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar calendar = Calendar.getInstance();
        int day = calendar.get(Calendar.DATE);
        calendar.set(Calendar.DATE, day + 1);
        return format.format(getNextDay());
    }

    /**
     * 获取当天/当周/当月剩余时间
     */
    public static long getExpireTime(String dateType) {
        long expireTime = 0;
        if (dateType.equals("DAY")) {
            expireTime = getTimeByDate(getNextDay()) - new Date().getTime();
        } else if (dateType.equals("WEEK")) {
            expireTime = getTimeByDate(getNextWeekFirstDay()) - new Date().getTime();
        } else if (dateType.equals("MONTH")) {
            expireTime = getTimeByDate(getNextMonthFirstDay()) - new Date().getTime();
        }
        return expireTime / 1000;
    }

    /**
     * 格林威治标准0时转化为北京时间（不带毫秒）
     *
     * @param time
     * @return
     */
    public static Date GMT2Beijing(String time) {
        DateFormat format1 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        format1.setTimeZone(TimeZone.getTimeZone("+0000"));
        try {
            return format1.parse(time);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 格林威治标准0时转化为北京时间（不带毫秒）
     *
     * @param time
     * @return
     */
    public static Date GMT2BeijingV2(String time) {
        DateFormat format1 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
        format1.setTimeZone(TimeZone.getTimeZone("+0000"));
        try {
            return format1.parse(time);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 北京时间转化为格林威治标准0时（不带毫秒）
     *
     * @param time
     * @return
     */
    public static String toGMT(Date time) {
        DateFormat format1 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        format1.setTimeZone(TimeZone.getTimeZone("+0000"));
        return format1.format(time);
    }

    /**
     * @param date
     * @return
     */
    public static Date getNextDayStartTime(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DATE) + 1, 0, 0, 0);
        return calendar.getTime();
    }

    public static Date getGMT() {
        DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
        return stringToDate(dateFormat.format(new Date()), "yyyy-MM-dd HH:mm:ss");
    }

    @Test
    public void test1() {
        System.out.println(dateToString(GMT2Beijing("2021-02-15T10:00:00Z")));
    }
}
