package com.bbk.sign;

import com.bbk.sign.constant.Constants;
import com.bbk.sign.constant.HttpHeader;
import com.bbk.sign.constant.SystemHeader;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * 签名类
 *
 * @author baigang
 * @date 2019-04-20 09:44
 * @since 1.0.0
 */
public class SignUtil {

    /**
     * 计算签名
     *
     * @param secret  秘钥
     * @param method  请求方法
     * @param path    路径
     * @param headers 头文件
     * @param bodys   请求体
     * @return
     */
    public static String sign(
            String secret,
            String method,
            String path,
            Map<String, String> headers,
            Map<String, String> bodys) {
        return sign(secret, method, path, headers, null, bodys);
    }

    /**
     * 计算签名
     *
     * @param secret APP密钥
     * @param method HttpMethod
     * @return 签名后的字符串
     */
    public static String sign(
            String secret,
            String method,
            String path,
            Map<String, String> headers,
            Map<String, String> querys,
            Map<String, String> bodys) {
        try {
            Mac hmacSha256 = Mac.getInstance(Constants.HMAC_SHA256);
            byte[] keyBytes = secret.getBytes(Constants.ENCODING);
            hmacSha256.init(new SecretKeySpec(keyBytes, 0, keyBytes.length,
                    Constants.HMAC_SHA256));

            return new String(Base64.encodeBase64(
                    hmacSha256.doFinal(
                            buildStringToSign(method, path, headers, querys,
                                    bodys)
                                    .getBytes(Constants.ENCODING))),
                    Constants.ENCODING);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 构建待签名字符串
     */
    private static String buildStringToSign(String method, String path,
                                            Map<String, String> headers,
                                            Map<String, String> querys,
                                            Map<String, String> bodys) {
        StringBuilder sb = new StringBuilder();

        sb.append(method.toUpperCase()).append(Constants.LF);
        if (null != headers) {
            if (null != headers.get(SystemHeader.X_CA_CONTENT_MD5)) {
                sb.append(headers.get(SystemHeader.X_CA_CONTENT_MD5));
            }
            sb.append(Constants.LF);
            if (null != headers.get(HttpHeader.HTTP_HEADER_CONTENT_TYPE)) {
                sb.append(headers.get(HttpHeader.HTTP_HEADER_CONTENT_TYPE));
            }
        }
        sb.append(Constants.LF);
        sb.append(buildHeaders(headers));
        sb.append(buildResource(path, querys, bodys));
//        System.out.println(sb.toString());
        return sb.toString();
    }

    /**
     * 构建待签名Path+Query+BODY
     *
     * @return 待签名
     */
    private static String buildResource(String path, Map<String, String> querys,
                                        Map<String, String> bodys) {
        StringBuilder sb = new StringBuilder();

        if (!StringUtils.isBlank(path)) {
            sb.append(path);
        }
        Map<String, String> sortMap = new TreeMap<String, String>();
        if (null != querys) {
            for (Map.Entry<String, String> query : querys.entrySet()) {
                if (!StringUtils.isBlank(query.getKey())) {
                    sortMap.put(query.getKey(), query.getValue());
                }
            }
        }

        if (null != bodys) {
            for (Map.Entry<String, String> body : bodys.entrySet()) {
                if (!StringUtils.isBlank(body.getKey())) {
                    sortMap.put(body.getKey(), body.getValue());
                }
            }
        }

        StringBuilder sbParam = new StringBuilder();
        for (Map.Entry<String, String> item : sortMap.entrySet()) {
            if (!StringUtils.isBlank(item.getKey())) {
                if (0 < sbParam.length()) {
                    sbParam.append(Constants.SPE3);
                }
                sbParam.append(item.getKey());
                if (!StringUtils.isBlank(item.getValue())) {
                    sbParam.append(Constants.SPE4).append(item.getValue());
                }
            }
        }
        if (0 < sbParam.length()) {
            sb.append(Constants.SPE5);
            sb.append(sbParam);
        }

        return sb.toString();
    }

    /**
     * 构建待签名Http头，由于目前s-和u-已经可以满足定制化需求，自定义的头参都以u-为前缀
     *
     * @param headers 请求中所有的Http头
     * @return 待签名Http头
     */
    private static String buildHeaders(Map<String, String> headers) {
        StringBuilder sb = new StringBuilder();

        if (null != headers) {
            headers.remove(SystemHeader.X_CA_SIGNATURE);
            headers.remove(SystemHeader.X_CA_CONTENT_MD5);
            headers.remove(HttpHeader.HTTP_HEADER_CONTENT_TYPE);
            Map<String, String> sortMap = new TreeMap<String, String>();
            sortMap.putAll(headers);
            StringBuilder signHeadersStringBuilder = new StringBuilder();
            for (Map.Entry<String, String> header : sortMap.entrySet()) {
                if (isHeaderToSign(header.getKey())) {
                    sb.append(header.getKey().toLowerCase());
                    sb.append(Constants.SPE2);
                    if (!StringUtils.isBlank(header.getValue())) {
                        sb.append(header.getValue());
                    }
                    sb.append(Constants.LF);
                    if (0 < signHeadersStringBuilder.length()) {
                        signHeadersStringBuilder.append(Constants.SPE1);
                    }
                    signHeadersStringBuilder.append(header.getKey());
                }
            }
            headers.put(SystemHeader.X_CA_SIGNATURE_HEADERS, signHeadersStringBuilder.toString());
        }

        return sb.toString();
    }

    /**
     * 构建待签名Http头,此方法为为了适应待签名头参数不固定，需要自定义时使用
     * 将所有待签名的头放入signHeaderPrefixList，暂时保留
     *
     * @param headers              请求中所有的Http头
     * @param signHeaderPrefixList 自定义参与签名Header前缀
     * @return 待签名Http头
     */
    private static String buildHeaders(Map<String, String> headers, List<String> signHeaderPrefixList) {
        StringBuilder sb = new StringBuilder();

        if (null != signHeaderPrefixList) {
            signHeaderPrefixList.remove(SystemHeader.X_CA_SIGNATURE);
            signHeaderPrefixList.remove(HttpHeader.HTTP_HEADER_ACCEPT);
            signHeaderPrefixList.remove(SystemHeader.X_CA_CONTENT_MD5);
            signHeaderPrefixList.remove(HttpHeader.HTTP_HEADER_CONTENT_TYPE);
            signHeaderPrefixList.remove(HttpHeader.HTTP_HEADER_DATE);
            Collections.sort(signHeaderPrefixList);
            if (null != headers) {
                Map<String, String> sortMap = new TreeMap<String, String>();
                sortMap.putAll(headers);
                StringBuilder signHeadersStringBuilder = new StringBuilder();
                for (Map.Entry<String, String> header : sortMap.entrySet()) {
                    if (isHeaderToSign(header.getKey(), signHeaderPrefixList)) {
                        sb.append(header.getKey());
                        sb.append(Constants.SPE2);
                        if (!StringUtils.isBlank(header.getValue())) {
                            sb.append(header.getValue());
                        }
                        sb.append(Constants.LF);
                        if (0 < signHeadersStringBuilder.length()) {
                            signHeadersStringBuilder.append(Constants.SPE1);
                        }
                        signHeadersStringBuilder.append(header.getKey());
                    }
                }
                headers.put(SystemHeader.X_CA_SIGNATURE_HEADERS, signHeadersStringBuilder.toString());
            }
        }

        return sb.toString();
    }

    /**
     * Http头是否参与签名 return
     */
    public static boolean isHeaderToSign(String headerName) {
        if (StringUtils.isBlank(headerName)) {
            return false;
        }

        if (SystemHeader.X_CA_SIGNATURE.equals(headerName)) {
            return false;
        }

        if (headerName.startsWith(Constants.CA_HEADER_TO_SIGN_PREFIX_SYSTEM)) {
            return true;
        }

        if (headerName.startsWith(Constants.CA_HEADER_TO_SIGN_PREFIX_USER)) {
            return true;
        }

        return false;
    }

    /**
     * Http头是否参与签名，用户自定义签名头 return
     */
    private static boolean isHeaderToSign(String headerName, List<String> signHeaderPrefixList) {
        if (StringUtils.isBlank(headerName)) {
            return false;
        }

        if (headerName.startsWith(Constants.CA_HEADER_TO_SIGN_PREFIX_SYSTEM)) {
            return true;
        }
        if (headerName.startsWith(Constants.CA_HEADER_TO_SIGN_PREFIX_USER)) {
            return true;
        }
        if (null != signHeaderPrefixList) {
            for (String signHeaderPrefix : signHeaderPrefixList) {
                if (headerName.equalsIgnoreCase(signHeaderPrefix)) {
                    return true;
                }
            }
        }
        return false;
    }
}
