package ellax.request.util;

import androidx.annotation.NonNull;

import ellax.base.IProgress;
import okhttp3.MediaType;
import okhttp3.RequestBody;
import okio.*;

import java.io.IOException;
import java.io.InputStream;

/**
 * Decorates an OkHttp request body to count the number of bytes written when writing it. Can
 * decorate any request body, but is most useful for tracking the upload inProgress of large
 * multipart requests.
 *
 * @author Leo Nikkilä
 */
public class ProgressRequestBody extends RequestBody {
    private RequestBody delegate;
    private IProgress mProgress;
    private String tag;

    public static RequestBody getRequestBody(RequestBody requestBody, IProgress progress, String tag) {
        if (progress == null)
            return requestBody;
        return new ProgressRequestBody(requestBody, progress, tag);
    }

    private ProgressRequestBody(RequestBody delegate, IProgress progress, String tag) {
        this.delegate = delegate;
        this.mProgress = progress;
        this.tag = tag;
    }

    @Override
    public MediaType contentType() {
        return delegate.contentType();
    }

    @Override
    public long contentLength() throws IOException {
        return delegate.contentLength();
    }

    @Override
    public void writeTo(BufferedSink sink) throws IOException {
        BufferedSink bufferedSink = Okio.buffer(new DelegateSink(sink));
        delegate.writeTo(bufferedSink);
        bufferedSink.flush();
    }

    private class DelegateSink extends ForwardingSink {
        private long bytesWritten = 0L;
        private long contentLength = -1L;

        public DelegateSink(Sink delegate) throws IOException {
            super(delegate);
            contentLength = contentLength();
        }

        @Override
        public void write(@NonNull Buffer source, long byteCount) throws IOException {
            super.write(source, byteCount);
            bytesWritten += byteCount;
            mProgress.update(bytesWritten, contentLength, bytesWritten / contentLength, tag);
        }
    }
}
