package ellax.request.response;

import androidx.annotation.NonNull;
import ellax.base.error.BusinessError;
import ellax.base.error.Error;
import ellax.base.error.ErrorFactory;
import ellax.base.error.ErrorTypes;
import ellax.request.Request;
import ellax.request.RequestConstants;
import ellax.request.RequestState;
import ellax.request.RequestStateManager;
import ellax.request.annotation.ReConst;
import ellax.request.cache.CacheCenter;
import ellax.request.intercept.IInterceptor;
import ellax.request.parser.ParseException;
import ellax.request.parser.Parser;
import ellax.request.util.DelegateStream;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;
import org.joor.Reflect;
import org.joor.ReflectException;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

/**
 * Created by dongdaqing on 2018/4/2.
 * 服务端返回的响应解析器
 */
public final class ResponseHandler implements Callback {
    private Parser mParser;//数据解析器
    private String mUniqueKey;
    private Map<String, String> mExtras;//用户传递的额外信息
    private Request<Object> mRequest;//数据回调接口
    private int tag;//请求标签
    private ResponseListener mResponseListener;

    private Error mError;

    public ResponseHandler(Parser parser, Request<Object> request, Map<String, String> extras, String uniqueKey, int tag, ResponseListener listener) {
        mParser = parser;
        mRequest = request;
        mExtras = extras;
        mUniqueKey = uniqueKey;
        mResponseListener = listener;
        this.tag = tag;
    }

    @Override
    public void onFailure(@NonNull Call call, @NonNull IOException e) {
        e.printStackTrace();
        mError = ErrorFactory.getError(e);
        dispatchError(call, "EXCEPTION", e.getClass().getSimpleName());
    }

    @Override
    public void onResponse(@NonNull Call call, @NonNull Response response) {
        if (response.isSuccessful()) {
            try {
                final long length = response.body().contentLength();
                InputStream is = DelegateStream.getStream(mRequest.getProgress(), response.body().byteStream(), length, null);
                Object data = mParser.parse(is, mExtras, length);
                if (data != null) {
                    dispatch(call, data);
                } else {
                    throw new ParseException(new Error(ErrorTypes.NO_DATA));
                }
            } catch (ParseException e) {
                mError = e.getError();
                dispatchError(call, "BUSINESS", String.valueOf(mError.getCode()));
            } catch (NullPointerException e) {
                mError = new Error(ErrorTypes.PARSE_FAILED);
                dispatchError(call, "BUSINESS", "");
            }
        } else {
            mError = ErrorFactory.getError(response.code());
            dispatchError(call, "HTTP", String.valueOf(response.code()));
        }
    }

    private void dispatch(Call call, Object data) {
        mRequest.mark(call.isCanceled());
        //标识数据来源：服务器
        mExtras.put(RequestConstants.DATA_SOURCE_KEY, RequestConstants.DATA_SOURCE_SERVER);
        mRequest.onRequestSuccess(data, mExtras);
        //请求结束
        mRequest.onFinish();
        RequestStateManager.stateChanged(mUniqueKey, RequestState.REQUEST_QUERY_SERVER_SUCCESS);
    }

    private void dispatchError(Call call, String type, String extra) {
        mRequest.mark(call.isCanceled());
        mError.setTag(tag);
        mError.setExtras(mExtras);
        if (!shouldIntercept())
            mRequest.onError(mError, mExtras);
        //请求结束
        mRequest.onFinish();
        RequestStateManager.stateChanged(mUniqueKey, RequestState.REQUEST_QUERY_SERVER_FAILURE);

        if (mResponseListener != null) {
            String method = mExtras.get("method");
            if (method != null && !method.isEmpty())
                mResponseListener.onResponseFailed(method, type, extra);
        }
    }

    /**
     * 是否拦截当前请求
     *
     * @return
     */
    private boolean shouldIntercept() {
        if (mError instanceof BusinessError) {
            try {
                Reflect reflect = Reflect.on(ReConst.R_PACKAGE + "." + ReConst.R_CLASS);
                if (reflect.call(ReConst.R_FUN_BOOLEAN, ((BusinessError) mError).getBusiCode()).get()) {
                    IInterceptor interceptor = reflect.call(ReConst.R_FUN_HANDLER).get();
                    interceptor.intercept(CacheCenter.getContext(), ((BusinessError) mError).getBusiCode());
                    return true;
                }
            } catch (ReflectException e) {
                return false;
            }
        }
        return false;
    }
}
