package ellax.request.parser;

import com.google.gson.*;
import com.google.gson.internal.$Gson$Types;
import ellax.base.Callback;
import ellax.base.error.BusinessError;
import ellax.base.error.Error;
import ellax.base.error.ErrorFactory;
import ellax.base.error.ErrorTypes;
import ellax.request.parser.annotation.*;
import ellax.request.util.IOUtil;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * Created by dongdaqing on 2018/4/2.
 * json解析器
 */
public class JsonParser implements Parser {

    private final static HashMap<Class, EnclosePack> mCache = new HashMap<>();
    /**
     * 服务端返回的数据都会有一个固定的格式：code，message，data
     * 这个encloseClass就是这个类，而下面的{@link #mType}就是data的具体类型
     * {@link #encloseClass}中的各个字段请使用注解进行标注，不然无法解析
     * {@link Code}
     * {@link Message}
     * {@link Data}
     * {@link Extra}
     * {@link BusiCode}
     */
    private Class encloseClass;
    private Type mType;
    private Callback<String> mCallback;

    public JsonParser(Callback<String> callback) {
        mCallback = callback;
    }

    public void setEncloseClass(Class encloseClass) {
        this.encloseClass = encloseClass;
    }

    public void setType(Type type) {
        mType = type;
    }

    public Type getType() {
        return mType;
    }

    public boolean isTypeSet() {
        return mType != null;
    }

    @Override
    public final Object parse(InputStream is, Map<String, String> extras, long length) throws ParseException {
        String response;
        try {
            response = IOUtil.readString(is);
        } catch (IOException e) {
            e.printStackTrace();
            //这样在上层处理时就不会漏掉这里的错误
            throw new ParseException(ErrorFactory.getNetworkError(e));
        }

        if (mCallback != null)
            mCallback.callback(response);

        final String name = $Gson$Types.getRawType(mType).getName();

        EnclosePack enclosePack;
        synchronized (mCache) {
            enclosePack = mCache.get(encloseClass);
            if (enclosePack == null) {
                enclosePack = EnclosePack.extract(encloseClass);
                mCache.put(encloseClass, enclosePack);
            }
        }
        try {
            JSONObject object = new JSONObject(response);
            //添加额外参数
            enclosePack.appendExtras(object, extras);
            if (enclosePack.isSuccess(object)) {
                //解析出具体数据并返回
                String data2Parse;
                if (encloseClass.getName().equals(name)) {
                    data2Parse = response;
                } else {
                    data2Parse = enclosePack.getData(object);
                }

                if (data2Parse == null || "null".equals(data2Parse) || "{}".equals(data2Parse)) {
                    throw new ParseException(new Error(ErrorTypes.NO_DATA));
                }

                //如果返回的内容类型是String就直接返回解析出来的原始数据
                if ("java.lang.String".equals(name))
                    return data2Parse;

                return parseData(data2Parse, mType);
            } else {
                //这里只是抛出业务错误，业务错误应该在业务逻辑层去解析和处理，这里只负责解析数据，不负责具体的业务
                throw new ParseException(new BusinessError(enclosePack.getBusiCode(object), enclosePack.getMessage(object), response));
            }
        } catch (JSONException e) {
            e.printStackTrace();
            throw new ParseException(new Error(ErrorTypes.PARSE_FAILED, "failed to parse response", response));
        }
    }

    /**
     * 解析数据
     */
    protected Object parseData(String data2Parse, Type type) throws ParseException {
        // TODO:bqf 2019-08-07 cache对象
        return new GsonBuilder().registerTypeAdapter(Long.class, new LongAdapter()).create().fromJson(data2Parse, type);
    }

    private static class LongAdapter implements JsonDeserializer<Long> {

        @Override
        public Long deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
            if (json == null) {
                return null;
            } else {
                try {
                    return json.getAsLong();
                } catch (Exception e) {
                    try {
                        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault());
                        return simpleDateFormat.parse(json.getAsString()).getTime();
                    } catch (Exception t) {
                        return null;
                    }
                }
            }
        }
    }

    private static class EnclosePack {
        private String code;
        private String busiCode;
        private String successCode;
        private String[] extraNames;
        private String[] extraKeys;
        private String[] message;
        private String[] data;

        private EnclosePack() {
            message = new String[0];
            data = new String[0];
            extraKeys = new String[0];
            extraNames = new String[0];
        }

        static EnclosePack extract(Class cls) {
            EnclosePack enclosePack = new EnclosePack();
            Field[] fields = cls.getDeclaredFields();
            for (Field field : fields) {
                Code code = field.getAnnotation(Code.class);
                if (code != null) {
                    enclosePack.code = field.getName();
                    enclosePack.successCode = code.code();
                    continue;
                }

                if (field.getAnnotation(Message.class) != null) {
                    enclosePack.message = enclosePack.add(enclosePack.message, field.getName());
                }

                if (field.getAnnotation(Data.class) != null) {
                    enclosePack.data = enclosePack.add(enclosePack.data, field.getName());
                }

                if (field.getAnnotation(BusiCode.class) != null) {
                    enclosePack.busiCode = field.getName();
                }

                Extra extra = field.getAnnotation(Extra.class);
                if (extra != null) {
                    enclosePack.extraKeys = enclosePack.add(enclosePack.extraKeys, field.getName());
                    enclosePack.extraNames = enclosePack.add(enclosePack.extraNames, extra.name());
                }
            }
            return enclosePack;
        }

        private String[] add(String[] dst, String value) {
            String[] tmp = new String[dst.length + 1];
            System.arraycopy(dst, 0, tmp, 0, dst.length);
            tmp[dst.length] = value;
            return tmp;
        }

        /**
         * 判断请求时否成功，这里判断的是业务的成功与否
         *
         * @throws ParseException 如果在返回的json中找不到对应的code，抛出业务异常
         */
        private boolean isSuccess(JSONObject object) throws ParseException {
            try {
                String returnCode = object.getString(code);
                return successCode.equals(returnCode);
            } catch (JSONException e) {
                String data = object.toString();
                throw new ParseException(new Error(ErrorTypes.PARSE_FAILED, "failed to find code: " + code + " in " + data, data));
            }
        }

        /**
         * 获取服务端返回的消息，一般是在返回业务逻辑失败时调用
         */
        private String getMessage(JSONObject object) {
            return getString(message, object);
        }

        /**
         * 获取真正的数据，一般是在返回业务逻辑成功时调用
         */
        private String getData(JSONObject object) {
            return getString(data, object);
        }

        private String getBusiCode(JSONObject object) {
            return object.optString(busiCode);
        }

        private void appendExtras(JSONObject object, Map<String, String> map) {
            for (int i = 0; i < extraKeys.length; i++) {
                try {
                    map.put(extraNames[i], object.getString(extraKeys[i]));
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }

        private String getString(String[] keys, JSONObject object) {
            if (keys.length == 0)
                return null;
            try {
                for (String key : keys) {
                    String ret = object.getString(key);
                    if (ret != null)
                        return ret;
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return null;
        }
    }
}
