package ellax.request.lifecycle;

import androidx.annotation.NonNull;
import ellax.base.IProgress;
import ellax.base.error.BusinessError;
import ellax.base.error.Error;
import ellax.base.lifecycle.LifecycleListener;
import ellax.base.lifecycle.LifecycleRegistry;
import ellax.request.Request;
import ellax.request.callback.IDataCallback;

import java.util.Map;

/**
 * Created by dongdaqing on 2018/4/3.
 * 与宿主生命周期绑定的{@link Request}，当宿主生命周期结束时会自动取消所有与该宿主绑定的{@link Request}
 */
public final class LifecycleRequest<T, A> implements IDataCallback<Object, A>, LifecycleListener {
    private Object mHost;
    private IDataCallback<T, A> mRequest;

    public LifecycleRequest(Object host, IDataCallback<T, A> request) {
        mHost = host;
        mRequest = request;
    }

    @Override
    public void mark(boolean cancel) {
        if (mRequest != null)
            mRequest.mark(cancel);
    }

    @Override
    public void onStart() {
        if (mRequest != null)
            mRequest.onStart();
    }

    @Override
    public void onRequestSuccess(@NonNull Object response, Map<String, String> extras) {
        if (mRequest != null)
            //noinspection unchecked
            mRequest.onRequestSuccess((T) response, extras);
    }

    @NonNull
    @Override
    public A transform(@NonNull Object data) {
        return null;
    }

    @Override
    public void onSuccess(@NonNull A data, Map<String, String> extras) {
        if (mRequest != null)
            mRequest.onSuccess(data, extras);
    }

    @Override
    public void onBusiError(BusinessError error, Map<String, String> extras) {

    }

    @Override
    public void onError(Error error, Map<String, String> extras) {
        if (mRequest != null)
            mRequest.onError(error, extras);
    }

    @Override
    public void onFinish() {
        //这里不回调mRequest的onFinish()方法，在UICallback里面有自己的一套逻辑
        removeFromHost();
    }

    @Override
    public IProgress getProgress() {
        if (mRequest != null)
            return mRequest.getProgress();
        return null;
    }

    @Override
    public Request getOriginalRequest() {
        if (mRequest != null)
            return mRequest.getOriginalRequest();
        return null;
    }

    @Override
    public void onLifecycleStart() {
        //见下面
    }

    @Override
    public void onLifecycleStop() {
        // FIXME: 2018/4/3 这里其实可以进一步优化，可以在这里暂停请求在onLifecycleStart时重新恢复
    }

    @Override
    public void onLifecycleDestroy() {
        if (mRequest != null)
            //宿主生命周期结束，取消当前请求
            mRequest.mark(true);
        //释放mRequest引用，避免内存泄漏
        mRequest = null;
        removeFromHost();
    }

    private void removeFromHost() {
        if (mHost != null) {
            LifecycleRegistry.unregister(mHost, this);
        }
        //释放引用，避免内存泄漏
        mHost = null;
    }
}
