package ellax.request.datasource;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;

import ellax.base.IProgress;
import ellax.base.error.Error;
import ellax.base.error.ErrorTypes;
import ellax.request.*;
import ellax.request.response.ResponseHandler;
import ellax.request.response.ResponseListener;
import ellax.request.util.ProgressRequestBody;
import ellax.request.util.UrlFormatter;
import okhttp3.*;

import java.io.File;
import java.util.Map;

/**
 * Created by dongdaqing on 2018/4/3.
 * 外部通过这个类来发起请求
 */
public class RemoteDataSource implements IDataSource {
    private Handler mHandler = new Handler(Looper.getMainLooper(), new Handler.Callback() {
        @Override
        public boolean handleMessage(Message msg) {
            Call call = (Call) msg.obj;
            call.cancel();
            return true;
        }
    });

    private ResponseListener mResponseListener;
    private NetworkClient mClient;

    public RemoteDataSource(boolean isShowLog) {
        mClient = new NetworkClient.Builder()
                .showLog(isShowLog)
                .build();
    }

    @Override
    public void setResponseListener(ResponseListener responseListener) {
        mResponseListener = responseListener;
    }

    @Override
    public void request(Params params, ellax.request.Request<Object> request) {
        if (mClient == null) {
            throw new RuntimeException("RemoteDataSource is not ready");
        }

        RequestStateManager.stateChanged(params.uniqueKey(), RequestState.REQUEST_QUERY_SERVER_START);

        ResponseHandler callback = new ResponseHandler(params.parser(), request, params.extras(), params.uniqueKey(), params.tag(), mResponseListener);
        okhttp3.Request okRequest = null;
        if (params.method() == HttpMethod.GET) {
            okRequest = getBuilder(params)
                    .url(UrlFormatter.appendParams(params.url(), params.params()))
                    .build();
        } else if (params.method() == HttpMethod.POST) {
            okRequest = getBuilder(params)
                    .url(params.url())
                    .post(getFormBody(params, params.files(), request.getProgress()))
                    .build();
        }

        if (okRequest != null) {
            Call call = mClient.newCall(okRequest);
            call.enqueue(callback);
            try {
                String to = params.extras().get(RequestConstants.REQUEST_CUSTOM_TIME_OUT);
                if (to != null) {
                    int timeout = Integer.parseInt(to);
                    if (timeout > 0 && timeout < mClient.getTimeoutSeconds()) {
                        Message message = mHandler.obtainMessage(0, call);
                        mHandler.sendMessageDelayed(message, timeout);
                    }
                }
            } catch (NumberFormatException e) {
                e.printStackTrace();
            }
        } else {
            request.onError(new Error(ErrorTypes.UNKNOWN_ERROR, "unknown method:" + params.method()), params.extras());
        }
    }

    private okhttp3.Request.Builder getBuilder(Params params) {
        okhttp3.Request.Builder builder = new okhttp3.Request.Builder();
        if (params.headers() != null) {
            Headers.Builder hb = new Headers.Builder();
            for (Map.Entry<String, String> entry : params.headers().entrySet()) {
                hb.add(entry.getKey(), entry.getValue());
            }
            builder.headers(hb.build());
        }
        builder.tag(params.tag());
        return builder;
    }

    /**
     * 构建用于上传的表单数据
     *
     * @param params
     * @param files
     * @param progress 进度监听接口
     * @return
     */
    private RequestBody getFormBody(Params params, Map<String, File> files, IProgress progress) {
        if (files.isEmpty()) {
            if(params.requestDataFormat() == Params.REQUEST_DATA_FORMAT_FORM) {
                FormBody.Builder builder = new FormBody.Builder();
                for (Map.Entry<String, String> entry : params.params().entrySet()) {
                    if (entry.getKey() != null && entry.getValue() != null)
                        builder.add(entry.getKey(), entry.getValue());
                }
                return builder.build();
            } else {
                RequestBody requestBody = FormBody.create(MediaType.parse("application/json; charset=utf-8"), params.jsonParams());
                return requestBody;
            }

        } else {
            MultipartBody.Builder builder = new MultipartBody.Builder();
            builder.setType(MultipartBody.FORM);
            for (Map.Entry<String, String> entry : params.params().entrySet()) {
                if (entry.getKey() != null && entry.getValue() != null)
                    builder.addFormDataPart(entry.getKey(), entry.getValue());
            }

            for (Map.Entry<String, File> entry : files.entrySet()) {
                // FIXME: 2018/4/3 这里可以提供一个对外接口，便于用户在上传之前对数据进行操作，比如说图片压缩，文件压缩
                File file = entry.getValue();
                builder.addFormDataPart(entry.getKey(), file.getName(), ProgressRequestBody.getRequestBody(RequestBody.create(MediaType.parse("application/octet-stream"), file), progress, file.getPath()));
            }
            return builder.build();
        }
    }

    @Override
    public void cancel(Object tag) {
        mClient.cancel(tag);
    }

    @Override
    public boolean haveSameTag(Object tag) {
        return mClient.haveSameTag(tag);
    }
}
