package ellax.request.callback;

import android.os.Looper;

import androidx.annotation.NonNull;

import ellax.base.IProgress;
import ellax.base.error.BusinessError;
import ellax.base.error.Error;
import ellax.base.error.ErrorTypes;
import ellax.base.helper.AppExecutors;
import ellax.request.Request;
import ellax.request.cache.CacheCenter;
import ellax.request.cache.CacheParam;

import java.util.List;
import java.util.Map;

/**
 * Created by dongdaqing on 2018/4/2.
 * 用于将数据Post到主线程，如果当前请求被取消，部分函数不会被会回调，这个callback不是给外部使用的
 */
public final class UICallback<T, A> implements IDataCallback<T, A> {
    //mark方法会跨线程调用
    private volatile boolean canceled;

    private IDataCallback<T, A> mCallback;
    private String mCacheKey;
    private CacheParam mCacheParam;

    private static ErrorInterceptor sErrorInterceptor;

    public UICallback(IDataCallback<T, A> callback, String cacheKey, CacheParam param) {
        mCacheKey = cacheKey;
        mCallback = callback;
        mCacheParam = param;
        //请求开始
        onStart();
    }

    @Override
    public void mark(boolean cancel) {
        //标记请求取消的两个地方：
        //1.请求绑定的宿主生命周期结束：生命周期监听器会分发Destroy事件，LifecycleRequest接收到状态之后会将请求标记为已取消，见LifecycleRequest
        //2.网络请求结果分发：网络请求完成（成功或者失败）之后，在分发结果之前会将当前请求的状态（是否已取消）标记到请求，见ResponseHandler
        //可能存在的情况，宿主结束了,但是网络请求还未完成，请求先标记为取消，网络请求完成的时候会重新标记状态，这时候如果不加控制就会将之前的状态覆盖掉，这样整个流程就会出问题
        //这里如果请求已经被取消，就无视任何的标记
        if (!canceled)
            canceled = cancel;
    }

    @Override
    public void onStart() {
        post(new Start(mCallback));
        if (canceled) {
            onFinish();
        }
    }

    @Override
    public void onRequestSuccess(@NonNull T response, Map<String, String> extras) {
        //列表数据为空也算做未请求到数据
        if (response instanceof List && ((List) response).isEmpty()) {
            onError(new Error(ErrorTypes.NO_DATA), extras);
            return;
        }

        //数据变换
        A a = mCallback.transform(response);
        if (mCacheKey != null) {
            //将数据写入缓存
            if (mCacheParam.writeToCache()) {
                //noinspection unchecked
                CacheCenter.update(mCacheKey, a, mCacheParam.clearBeforeUpdate());
            }
            CacheCenter.markNonDirty(mCacheKey);
        }
        onSuccess(a, extras);
    }

    @NonNull
    @Override
    public A transform(@NonNull T data) {
        return null;
    }

    @Override
    public void onSuccess(@NonNull A data, Map<String, String> extras) {
        postSuccess(data, extras);
    }

    @Override
    public void onBusiError(BusinessError error, Map<String, String> extras) {
        //UICallback中此方法不会被调用
    }

    @Override
    public void onError(Error error, Map<String, String> extras) {
        postError(error, extras);
    }

    @Override
    public void onFinish() {
        //即使请求被取消，onFinish函数也会被调用
        AppExecutors.mainThread(new Finish(mCallback));
    }

    @Override
    public IProgress getProgress() {
        return mCallback.getProgress();
    }

    @Override
    public Request getOriginalRequest() {
        return mCallback.getOriginalRequest();
    }

    private void postSuccess(A data, Map<String, String> extras) {
        post(new Success<>(mCallback, data, extras));
        //请求结束
        onFinish();
    }

    private void postError(Error error, Map<String, String> extras) {
        //有几个业务逻辑错误是全局的，需要拦截处理
        if (sErrorInterceptor == null || !sErrorInterceptor.intercept(error))
            post(new Fail(error, extras, mCallback));
        //请求结束
        onFinish();
    }

    private void post(Runnable runnable) {
        if (!canceled) {
            if (Looper.getMainLooper() != Looper.myLooper())
                AppExecutors.mainThread(runnable);
            else
                runnable.run();
        }
    }

    private static class Start implements Runnable {
        private IDataCallback mCallback;

        public Start(IDataCallback callback) {
            mCallback = callback;
        }

        @Override
        public void run() {
            mCallback.onStart();
        }
    }

    private static class Success<T, A> implements Runnable {
        private IDataCallback<T, A> mCallback;
        private Map<String, String> mExtras;
        private A data;

        public Success(IDataCallback<T, A> callback, A data, Map<String, String> extras) {
            mCallback = callback;
            mExtras = extras;
            this.data = data;
        }

        @Override
        public void run() {
            mCallback.onSuccess(data, mExtras);
        }
    }

    private static class Fail implements Runnable {
        private Error mError;
        private Map<String, String> extras;
        private IDataCallback mCallback;

        public Fail(Error error, Map<String, String> extras, IDataCallback callback) {
            mError = error;
            this.extras = extras;
            mCallback = callback;
        }

        @Override
        public void run() {
            if (mError instanceof BusinessError)
                mCallback.onBusiError((BusinessError) mError, extras);
            else
                mCallback.onError(mError, extras);
        }
    }

    private static class Finish implements Runnable {
        private IDataCallback mCallback;

        public Finish(IDataCallback callback) {
            mCallback = callback;
        }

        @Override
        public void run() {
            mCallback.onFinish();
        }
    }

    public static void setErrorInterceptor(ErrorInterceptor errorInterceptor) {
        sErrorInterceptor = errorInterceptor;
    }
}
