package ellax.request.callback;

import androidx.annotation.CallSuper;
import androidx.annotation.NonNull;
import ellax.base.EllaView;
import ellax.base.IProgress;
import ellax.base.error.BusinessError;
import ellax.base.error.Error;
import ellax.request.Request;

import java.util.Map;

/**
 * created by dongdaqing 19-3-7 下午4:19
 * <p>
 * UI层回调接口，大部分情况下使用{@link SimpleCallback}就足够了
 */
public abstract class TransferCallback<T, A> implements IDataCallback<T, A> {
    private EllaView mEllaView;
    private IProgress mProgress;

    public TransferCallback() {
    }

    public TransferCallback(EllaView ellaView) {
        this(ellaView, ellaView);
    }

    public TransferCallback(EllaView ellaView, IProgress progress) {
        mEllaView = ellaView;
        mProgress = progress;
    }

    @Override
    public final void mark(boolean cancel) {
        //这个函数不会被调用，重写这个函数没意义，so。。。
    }

    @CallSuper
    @Override
    public void onStart() {
        if (getProgress() != null)
            getProgress().showProgress();
    }

    @Override
    public void onSuccess(@NonNull A data, Map<String, String> extras) {
        //重写这个方法来更新UI
    }

    @Override
    public final void onRequestSuccess(@NonNull T response, Map<String, String> extras) {
        //这个接口不会在UI层回调
    }

    @Override
    public void onBusiError(BusinessError error, Map<String, String> extras) {
        //处理业务逻辑错误，默认当做通用错误处理
        onError(error, extras);
    }

    @Override
    public void onError(Error error, Map<String, String> extras) {
        //处理其他错误
        if (mEllaView != null)
            mEllaView.handleError(error);
    }

    @CallSuper
    @Override
    public void onFinish() {
        stopProgress();
    }

    @Override
    public IProgress getProgress() {
        return mProgress;
    }

    @Override
    public Request getOriginalRequest() {
        return this;
    }

    private void stopProgress() {
        if (mProgress != null)
            mProgress.stopProgress();

        //避免重复调用
        mProgress = null;
    }
}
