package ellax.request.callback;

import androidx.annotation.NonNull;
import ellax.base.error.BusinessError;
import ellax.base.error.Error;
import ellax.request.Request;

import java.util.Map;

/**
 * Created by dongdaqing on 2018/4/2.
 * 业务层的数据回调接口
 * <p>
 * 为什么不把这下面的几个函数放到{@link Request}里面去？
 * 私以为下面这部分涉及到了业务处理，但抽象出来的请求过程应是与业务逻辑无关的，
 * 网络部分处理的时候只关心基本的操作（开始、结束、取消、成功、失败），要细分操作，
 * 可以在业务逻辑层面去做，保持功能的简洁
 */
public interface IDataCallback<T, A> extends Request<T> {

    /**
     * 以上两个泛型中 T 表示服务端返回的数据的类型，即我们用什么类型来接收服务端传过来的数据
     *             A 表示发起请求方需要的数据类型，即调用方最终会获得的数据类型
     *
     * 这个方法的意义在于，可以通过标准的数据解析过程将服务端的数据解析出来，然后再将数据进行二次变换。
     * 在很多的例子中都存在服务端给我的数据和我要用来做作展示的数据有差异，需要将数据进行二次封装处理，这里直接提供一个
     * 在子线程进行数据封装的方法
     * @param data
     * @return
     */
    @NonNull A transform(@NonNull T data);

    /**
     * 请求成功
     *
     * @param data 经过{@link #transform(Object)}的数据
     * @param extras   extra info for this request
     */
    void onSuccess(@NonNull A data, Map<String, String> extras);


    /**
     * 业务错误的处理函数，这里将业务错误和其他错误分离开来是根据实际的情况来决定的，
     * 如果不拆分那么就必须在每个要处理业务错误的地方都要写if-else语句，多了都是冗余代码，所以还是拆分的好
     *
     * @param error
     * @param extras
     */
    void onBusiError(BusinessError error, Map<String, String> extras);

    /**
     * 除开业务错误的其他错误，业务错误不会调用这个函数
     *
     * @param error
     * @param extras
     */
    void onError(Error error, Map<String, String> extras);
}
