package ellax.request;


import java.io.File;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;

import ellax.base.Callback;
import ellax.request.cache.CacheParam;
import ellax.request.parser.JsonParser;
import ellax.request.parser.Parser;

/**
 * Created by dongdaqing on 2018/4/2.
 */
public class RequestParams implements Params {
    private int tag;
    private String url;//请求地址
    private HttpMethod mMethod;//请求方法
    //额外参数，会在整个请求的生命周期中流转,这里可以放一些请求相关的参数，
    // 比如文件下载的本地地址，列表请求是不是刷新操作等这些描述请求状态的参数
    private Map<String, String> mExtras;
    private Map<String, String> mParams;//发起请求的参数
    private String mJsonParams; //发起请求的json参数
    private Map<String, String> mHeaders;//添加到请求头里面的参数
    private Map<String, File> mFiles;//文件上传用
    private CacheParam mCacheParam;
    private String uniqueKey;//能够唯一标识请求的key值
    private Parser mParser;//响应解析器
    private int transitionKey;

    /**
     * 请求报文格式，支持json或form表单
     */
    private int requestDataFormat = REQUEST_DATA_FORMAT_FORM;

    private RequestParams() {
        this(REQUEST_DATA_FORMAT_FORM);
    }

    private RequestParams(@RequestDataFormat int requestDataFormat) {
        if (requestDataFormat == REQUEST_DATA_FORMAT_FORM) {
            mExtras = new HashMap<>(0);
            mParams = new HashMap<>(0);
            mHeaders = new HashMap<>(0);
            mFiles = new HashMap<>(0);
            this.requestDataFormat = REQUEST_DATA_FORMAT_FORM;
        } else if (requestDataFormat == REQUEST_DATA_FORMAT_JSON) {
            mExtras = new HashMap<>(0);
            mHeaders = new HashMap<>(0);
            mFiles = new HashMap<>(0);
            this.requestDataFormat = REQUEST_DATA_FORMAT_JSON;
        } else {
            throw new IllegalArgumentException("invalid request data format");
        }
    }

    @Override
    public String url() {
        return url;
    }

    @Override
    public Map<String, String> extras() {
        return mExtras;
    }

    @Override
    public Map<String, String> params() {
        return mParams;
    }

    @Override
    public String jsonParams() {
        return mJsonParams;
    }

    @Override
    public Map<String, String> headers() {
        return mHeaders;
    }

    @Override
    public Map<String, File> files() {
        return mFiles;
    }

    @Override
    public HttpMethod method() {
        return mMethod;
    }

    @Override
    public Parser parser() {
        return mParser;
    }

    @Override
    public CacheParam cache() {
        return mCacheParam;
    }

    @Override
    public int tag() {
        return tag;
    }

    @Override
    public String uniqueKey() {
        return uniqueKey;
    }

    @Override
    public int transitionKey() {
        return transitionKey;
    }

    @Override
    public int requestDataFormat() {
        return requestDataFormat;
    }

    public void param(String key, String value) {
        if (requestDataFormat != REQUEST_DATA_FORMAT_FORM) {
            throw new RuntimeException("your data format is not form, should use jsonParam()!");
        }
        if (value != null)
            mParams.put(key, value);
    }

    public void jsonParam(String jsonStr) {
        if (requestDataFormat != REQUEST_DATA_FORMAT_JSON) {
            throw new RuntimeException("your data format is not json, should use param()!");
        }
        mJsonParams = jsonStr;
    }

    public void setParams(Map<String, String> map) {
        mParams.putAll(map);
    }

    public static class Builder {
        private RequestParams mParams;
        private Class mEncloseClass;
        private Type mType;
        private Callback<String> mCallback;

        public Builder() {
            this(REQUEST_DATA_FORMAT_FORM);
        }

        public Builder(@RequestDataFormat int dataFormat) {
            if (dataFormat != REQUEST_DATA_FORMAT_FORM && dataFormat != REQUEST_DATA_FORMAT_JSON) {
                throw new IllegalArgumentException("invalid request data format");
            }
            mParams = new RequestParams(dataFormat);
        }

        public Builder(RequestParams params) {
            mParams = params;
        }

        public Builder url(String url) {
            mParams.url = url;
            return this;
        }

        /**
         * 具体意义见{@link JsonParser}
         *
         * @param cls
         * @return
         */
        public Builder setEncloseClass(Class cls) {
            mEncloseClass = cls;
            return this;
        }

        /**
         * 如果需要封装RequestParams，请通过这个函数传入具体类型，这个函数
         *
         * @param type
         */
        public Builder setType(Type type) {
            mType = type;
            return this;
        }

        /**
         * json parser在服务端结果解析成功之后的回调，可以再这个回调里面进行日志打印或者记录操作
         *
         * @param callback
         * @return
         */
        public Builder setCallback(Callback<String> callback) {
            mCallback = callback;
            return this;
        }

        public Builder parser(Parser parser) {
            mParams.mParser = parser;
            return this;
        }

        public Builder tag(int tag) {
            mParams.tag = tag;
            return this;
        }

        public Builder method(HttpMethod method) {
            mParams.mMethod = method;
            return this;
        }

        public Builder param(String key, String value) {
            mParams.param(key, value);
            return this;
        }

        public Builder params(Map<String, String> params) {
            mParams.setParams(params);
            return this;
        }

        public Builder jsonParams(String jsonStr) {
            mParams.jsonParam(jsonStr);
            return this;
        }

        public Builder extra(String key, String value) {
            mParams.mExtras.put(key, value);
            return this;
        }

        public Builder extras(Map<String, String> params) {
            mParams.mExtras.putAll(params);
            return this;
        }

        public Builder file(String key, String value) {
            if (value != null)
                mParams.mFiles.put(key, new File(value));
            return this;
        }

        public Builder header(String key, String value) {
            if (value != null)
                mParams.mHeaders.put(key, value);
            return this;
        }

        public Builder header(Map<String, String> headerMap) {
            mParams.mHeaders.putAll(headerMap);
            return this;
        }

        public Builder cache(CacheParam param) {
            mParams.mCacheParam = param;
            return this;
        }

        public Builder uniqueKey(String key) {
            mParams.uniqueKey = key;
            return this;
        }

        public Builder bindTransition(int hostKey) {
            mParams.transitionKey = hostKey;
            return this;
        }

        public RequestParams get() {
            mParams.mMethod = HttpMethod.GET;
            return build();
        }

        public RequestParams post() {
            mParams.mMethod = HttpMethod.POST;
            return build();
        }

        public RequestParams build() {
            checkParser();
            return mParams;
        }

        protected void checkParser() {
            if (mParams.mParser == null) {
                mParams.mParser = new JsonParser(mCallback);
            }

            if (mParams.mParser instanceof JsonParser) {
                ((JsonParser) mParams.mParser).setEncloseClass(mEncloseClass);
                ((JsonParser) mParams.mParser).setType(mType);
            }
        }
    }
}
