package ellax.request;

import android.text.TextUtils;
import android.util.Log;

import androidx.viewbinding.BuildConfig;

import ellax.request.intercept.HttpLoggingInterceptor;
import okhttp3.Call;
import okhttp3.ConnectionSpec;
import okhttp3.OkHttpClient;
import okhttp3.Request;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.X509TrustManager;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Created by dongdaqing on 2018/4/2.
 */
public class NetworkClient {
    private OkHttpClient mClient;

    private NetworkClient(Builder builder) {
        OkHttpClient.Builder okHttpBuilder = new OkHttpClient.Builder()
                .connectTimeout(builder.connectTimeout, TimeUnit.SECONDS)
                .readTimeout(builder.readTimeout, TimeUnit.SECONDS)
                .writeTimeout(builder.writeTimeout, TimeUnit.SECONDS);

        if (builder.mSSLSocketFactory != null && builder.mX509TrustManager != null) {
            okHttpBuilder.sslSocketFactory(builder.mSSLSocketFactory, builder.mX509TrustManager);
            okHttpBuilder.connectionSpecs(Collections.singletonList(ConnectionSpec.MODERN_TLS));
        }

        if (builder.isShowLog) {
            HttpLoggingInterceptor interceptor = new HttpLoggingInterceptor(message -> {
                if (TextUtils.isEmpty(message)) {
                    return;
                }
                int len = message.length();
                int readLen = 0;
                while (readLen < len) {
                    int end = readLen + 4000;
                    if (readLen + 4000 > len) {
                        end = len;
                    }
                    Log.d("HttpLoggingInterceptor", message.substring(readLen, end));
                    readLen = end;
                }
            });
            interceptor.setLevel(HttpLoggingInterceptor.Level.BODY);
            okHttpBuilder.addInterceptor(interceptor);
        }

        if (builder.mHostnameVerifier != null)
            okHttpBuilder.hostnameVerifier(builder.mHostnameVerifier);

        mClient = okHttpBuilder.build();
    }

    public Call newCall(Request request) {
        return mClient.newCall(request);
    }

    public void cancelAll() {
        mClient.dispatcher().cancelAll();
    }

    public void cancel(Object tag) {
        if (tag == null) return;

        cancel(tag, mClient.dispatcher().runningCalls());
        cancel(tag, mClient.dispatcher().queuedCalls());
    }

    public int getTimeoutSeconds() {
        return mClient.readTimeoutMillis() / 1000;
    }

    private void cancel(Object tag, List<Call> calls) {
        for (Call call : calls) {
            if (tag.equals(call.request().tag())) {
                call.cancel();
            }
        }
    }

    public boolean haveSameTag(Object tag) {
        for (Call call : mClient.dispatcher().runningCalls()) {
            if (tag.equals(call.request().tag())) {
                return true;
            }
        }
        for (Call call : mClient.dispatcher().queuedCalls()) {
            if (tag.equals(call.request().tag())) {
                return true;
            }
        }
        return false;
    }

    public static class Builder {
        private static final int DEFAULT_CONNECT_TIMEOUT = 10;
        private static final int DEFAULT_READ_TIMEOUT = 30;
        private static final int DEFAULT_WRITE_TIMEOUT = 30;

        private int writeTimeout = DEFAULT_WRITE_TIMEOUT;
        private int readTimeout = DEFAULT_READ_TIMEOUT;
        private int connectTimeout = DEFAULT_CONNECT_TIMEOUT;
        private SSLSocketFactory mSSLSocketFactory;
        private X509TrustManager mX509TrustManager;
        private HostnameVerifier mHostnameVerifier;
        private boolean isShowLog;

        public Builder writeTimeout(int seconds) {
            writeTimeout = seconds;
            return this;
        }

        public Builder readTimeout(int seconds) {
            readTimeout = seconds;
            return this;
        }

        public Builder connectTimeout(int seconds) {
            readTimeout = seconds;
            return this;
        }

        public Builder sSLSocketFactory(SSLSocketFactory SSLSocketFactory) {
            mSSLSocketFactory = SSLSocketFactory;
            return this;
        }

        public Builder x509TrustManager(X509TrustManager x509TrustManager) {
            mX509TrustManager = x509TrustManager;
            return this;
        }

        public Builder hostnameVerifier(HostnameVerifier hostnameVerifier) {
            mHostnameVerifier = hostnameVerifier;
            return this;
        }

        public Builder showLog(boolean isShowLog) {
            this.isShowLog = isShowLog;
            return this;
        }

        public NetworkClient build() {
            return new NetworkClient(this);
        }
    }
}
