package ellax.request;

import androidx.annotation.Nullable;
import androidx.viewbinding.BuildConfig;

import com.google.gson.reflect.TypeToken;
import ellax.base.Callback;
import ellax.base.lifecycle.LifecycleRegistry;
import ellax.request.cache.Cache;
import ellax.request.cache.CacheCenter;
import ellax.request.cache.CacheParam;
import ellax.request.cache.LifecycleAwareCache;
import ellax.request.callback.IDataCallback;
import ellax.request.callback.UICallback;
import ellax.request.datasource.IDataSource;
import ellax.request.datasource.RemoteDataSource;
import ellax.request.lifecycle.LifecycleRequest;
import ellax.request.parser.JsonParser;
import ellax.request.parser.Parser;
import ellax.request.response.ResponseListener;
import ellax.request.util.TokenHelper;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.WildcardType;
import java.util.List;

/**
 * created by dongdaqing 19-3-4 下午6:13
 */
public class EllaTask {
    private static IDataSource sDataSource;

    public static <T, A> void fetchData(@Nullable Object host, Params params, IDataCallback<T, A> callback) {

        if (params.transitionKey() != 0)
            TransitionHelper.bindWithRequest(params.transitionKey(), params.uniqueKey());

        RequestStateManager.stateChanged(params.uniqueKey(), RequestState.REQUEST_START);
        if (params.tag() != 0 && RequestConstants.shouldAutoCancelPendingRequest(params.extras())) {
            cancel(params.tag());
        }

        if (params.tag() != 0 && RequestConstants.shouldAbortOnSameTag(params.extras()) && getDataSource().haveSameTag(params.tag()))
            return;

        checkType(params.parser(), callback);

        /**
         * 只要设置了缓存参数就认为是需要缓存的，如果未注册缓存处理器（当前接口），这里会自动注册一个和当前{@link host}生命周期绑定的缓存，
         * {@link host}生命周期结束时会自动取消缓存（主要是销毁内存缓存）
         */
        CacheParam cacheParam = params.cache();
        String cacheKey = cacheParam == null ? null : cacheParam.getCacheKey(params);

        //这里使用LifecycleRequest作为代理来与宿主的生命周期绑定
        IDataCallback<T, A> request = new UICallback<>(callback, cacheKey, params.cache());
        LifecycleRequest<T, A> lifecycleWrapper = new LifecycleRequest<>(host, request);
        LifecycleRegistry.register(host, lifecycleWrapper);

        if (cacheKey != null) {
            if (!CacheCenter.cacheEnabledForKey(cacheKey)) {
                //如果参数中设置了缓存数据类型那就直接使用，若没有，就获取传入的callback接口中的第二个泛型参数
                Type type = cacheParam.getType() == null ? TokenHelper.getTargetType(callback.getClass()) : cacheParam.getType();

                if (type == null)
                    throw new IllegalArgumentException("can not get type for cache");

                int cacheType = cacheParam.getCacheType();
                Type ct = type;

                if (cacheType == CacheParam.CACHE_TYPE_AUTO) {
                    Type rawType = TypeToken.get(type).getRawType();
                    if (List.class.isAssignableFrom((Class<?>) rawType)) {
                        cacheType = CacheParam.CACHE_TYPE_LIST;
                        ParameterizedType parameterizedType = (ParameterizedType) type;
                        ct = parameterizedType.getActualTypeArguments()[0];

                        if (ct instanceof WildcardType) {
                            ct = ((WildcardType) ct).getUpperBounds()[0];
                        }

                    } else {
                        cacheType = CacheParam.CACHE_TYPE_NORMAL;
                    }
                }

                Cache cache = CacheCenter.createCache(ct, cacheType == CacheParam.CACHE_TYPE_LIST);
                LifecycleRegistry.register(host, new LifecycleAwareCache(cacheKey, cache));
                CacheCenter.enableCache(cacheKey, cache);
            }

            if (cacheParam.readFromCache()) {
                CacheCenter.query(params.uniqueKey(), cacheKey, new QueryCallback<>(lifecycleWrapper, params));
            } else {
                getDataSource().request(params, lifecycleWrapper);
            }
        } else {
            getDataSource().request(params, lifecycleWrapper);
        }
    }

    private static IDataSource getDataSource() {
        if (sDataSource == null) {
            sDataSource = new RemoteDataSource(BuildConfig.DEBUG);
        }

        return sDataSource;
    }

    public static void setDataSource(IDataSource dataSource){
        sDataSource = dataSource;
    }

    public static void setResponseListener(ResponseListener responseListener) {
        getDataSource().setResponseListener(responseListener);
    }

    /**
     * 检查JsonParser中的Type是否已经设定，如果没有就通过泛型获取
     *
     * @param parser
     * @param request
     */
    private static <T> void checkType(Parser parser, Request<T> request) {
        if (parser instanceof JsonParser) {
            JsonParser jsonParser = (JsonParser) parser;
            if (!jsonParser.isTypeSet()) {
                jsonParser.setType(TokenHelper.getSourceType(request.getClass()));
            }
        }
    }

    public static void cancel(int tag) {
        if (sDataSource != null)
            getDataSource().cancel(tag);
    }

    private static class QueryCallback<T, A> implements Callback<A> {

        private LifecycleRequest<T, A> mRequest;
        private Params mParams;

        private QueryCallback(LifecycleRequest<T, A> request, Params params) {
            mRequest = request;
            mParams = params;
        }

        @Override
        public void callback(A data) {
            RequestStateManager.stateChanged(mParams.uniqueKey(), RequestState.REQUEST_QUERY_CACHE_END);
            CacheParam cacheParam = mParams.cache();
            String key = cacheParam.getCacheKey(mParams);
            if (data != null) {
                boolean dirtyCache = CacheCenter.isDirtyCache(key);
                if (!dirtyCache || !cacheParam.shouldDiscardDirtyCache()) {
                    mParams.extras().put(RequestConstants.DATA_SOURCE_KEY, RequestConstants.DATA_SOURCE_CACHE);
                    mRequest.onSuccess(data, mParams.extras());
                }

                if (dirtyCache) {
                    getDataSource().request(mParams, mRequest);
                }
            } else {
                getDataSource().request(mParams, mRequest);
            }
        }
    }
}
