package ellax.base.helper;

import android.app.Activity;
import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.os.Build;

import androidx.annotation.AnimRes;
import androidx.annotation.ColorInt;
import androidx.annotation.ColorRes;
import androidx.annotation.DrawableRes;
import androidx.annotation.FloatRange;
import androidx.annotation.IdRes;
import androidx.annotation.IntDef;
import androidx.annotation.StringRes;
import androidx.fragment.app.FragmentTransaction;
import androidx.core.content.ContextCompat;
import androidx.core.view.ViewCompat;
import androidx.core.widget.TextViewCompat;
import androidx.appcompat.content.res.AppCompatResources;
import android.text.InputType;
import android.util.SparseArray;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.*;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * @author dongdaqing 2018-05-30 下午5:53
 * 一个简化初始化代码的帮助类
 */
public class ViewHelper {
    public static final int MARGIN_ALL = 0;
    public static final int MARGIN_LEFT = 1;
    public static final int MARGIN_RIGHT = 2;
    public static final int MARGIN_TOP = 4;
    public static final int MARGIN_BOTTOM = 8;

    public static final int MARGIN_HORIZONTAL = MARGIN_LEFT | MARGIN_RIGHT;
    public static final int MARGIN_VERTICAL = MARGIN_TOP | MARGIN_BOTTOM;

    @IntDef({MARGIN_ALL,
            MARGIN_TOP,
            MARGIN_BOTTOM,
            MARGIN_LEFT,
            MARGIN_RIGHT,
            MARGIN_HORIZONTAL,
            MARGIN_VERTICAL})
    @Retention(RetentionPolicy.SOURCE)
    public @interface MarginPosition {
    }

    private SparseArray<View> mViewArray;
    private View mRootView;

    public ViewHelper(Activity activity) {
        this(activity.getWindow().getDecorView().getRootView().findViewById(android.R.id.content));
    }

    public ViewHelper(View rootView) {
        if (rootView == null)
            throw new IllegalArgumentException("rootView is null");

        mRootView = rootView;
        mViewArray = new SparseArray<>();
    }

    public ViewHelper(Context context, int layout) {
        mRootView = LayoutInflater.from(context).inflate(layout, null);
        mViewArray = new SparseArray<>();
    }

    public void setText(@IdRes int id, CharSequence sequence) {
        TextView textView = getView(id);
        if (textView != null) {
            textView.setText(sequence);
            if (textView instanceof EditText) {
                ((EditText) textView).setSelection(textView.getText().length());
            }
        }
    }

    public void setHint(@IdRes int id, CharSequence sequence) {
        TextView textView = getView(id);
        if (textView != null) {
            textView.setHint(sequence);
        }
    }

    public void setHintColor(@IdRes int id, @ColorInt int color) {
        TextView textView = getView(id);
        if (textView != null) {
            textView.setHintTextColor(color);
        }
    }

    public void setText(@IdRes int id, @StringRes int sequence) {
        TextView textView = getView(id);
        if (textView != null)
            textView.setText(sequence);
    }

    public void setTextSize(@IdRes int id, float px) {
        setTextSize(id, px, TypedValue.COMPLEX_UNIT_PX);
    }

    public void setTextSize(@IdRes int id, float size, int unit) {
        TextView textView = getView(id);
        if (textView != null)
            textView.setTextSize(unit, size);
    }

    public void setTextColor(@IdRes int id, @ColorInt int color) {
        TextView textView = getView(id);
        if (textView != null)
            textView.setTextColor(color);
    }

    public void setTextColorRes(@IdRes int id, @ColorRes int color) {
        TextView textView = getView(id);
        if (textView != null)
            textView.setTextColor(ContextCompat.getColor(mRootView.getContext(), color));
    }

    public void setTextColor(@IdRes int id, String color) {
        TextView textView = getView(id);
        if (textView != null)
            textView.setTextColor(Color.parseColor(color));
    }

    public void setTextGravity(@IdRes int id, @ColorInt int gravity) {
        TextView textView = getView(id);
        if (textView != null)
            textView.setGravity(gravity);
    }

    public void moveCursor2End(@IdRes int id) {
        EditText editText = getView(id);
        if (editText != null)
            editText.setSelection(editText.getText().length());
    }

    public void setCompoundDrawableStart(@IdRes int id, @DrawableRes int drawable) {
        TextView textView = getView(id);
        if (textView != null)
            TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(textView, getDrawable(drawable), null, null, null);
    }

    public void setCompoundDrawableRight(@IdRes int id, @DrawableRes int drawable) {
        TextView textView = getView(id);
        if (textView != null) {
            TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(textView, null, null, getDrawable(drawable), null);
        }
    }

    public void setCompoundDrawableTop(@IdRes int id, @DrawableRes int drawable) {
        TextView textView = getView(id);
        if (textView != null)
            TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(textView, null, getDrawable(drawable), null, null);
    }

    public void setCompoundDrawableStart(@IdRes int id, Drawable drawable) {
        TextView textView = getView(id);
        if (textView != null)
            TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(textView, drawable, null, null, null);
    }

    public void setCompoundDrawableRight(@IdRes int id, Drawable drawable) {
        TextView textView = getView(id);
        if (textView != null) {
            TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(textView, null, null, drawable, null);
        }
    }

    public void setCompoundDrawableTop(@IdRes int id, Drawable drawable) {
        TextView textView = getView(id);
        if (textView != null)
            TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(textView, null, drawable, null, null);
    }

    public void setProgress(@IdRes int id, int progress) {
        ProgressBar progressBar = getView(id);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N)
            progressBar.setProgress(progress, true);
        else
            progressBar.setProgress(progress);
    }

    public String getText(@IdRes int id) {
        TextView textView = getView(id);
        if (textView instanceof EditText) {
            return textView.getText().toString().trim();
        }
        return textView.getText().toString();
    }

    public float getTextSize(@IdRes int id) {
        TextView textView = getView(id);
        return textView.getTextSize();
    }

    public void setChecked(@IdRes int id, boolean checked) {
        Checkable checkable = getView(id);
        if (checkable != null)
            checkable.setChecked(checked);
    }

    public void setCheckedListener(@IdRes int id, CompoundButton.OnCheckedChangeListener listener) {
        CompoundButton compoundButton = getView(id);
        if (compoundButton != null)
            compoundButton.setOnCheckedChangeListener(listener);
    }

    public boolean isChecked(@IdRes int id) {
        Checkable checkable = getView(id);
        return checkable.isChecked();
    }

    public void setSelected(@IdRes int id, boolean selected) {
        View view = getView(id);
        if (view != null)
            view.setSelected(selected);
    }

    public boolean isSelected(@IdRes int id) {
        View view = getView(id);
        return view.isSelected();
    }

    public void setImage(@IdRes int id, @DrawableRes int drawable) {
        ImageView iv = getView(id);
        if (iv != null) {
            iv.setImageDrawable(getDrawable(drawable));
        }
    }

    public void setImage(@IdRes int id, Drawable drawable) {
        ImageView iv = getView(id);
        if (iv != null) {
            iv.setImageDrawable(drawable);
        }
    }

    public void setBackground(@IdRes int id, @DrawableRes int drawable) {
        View iv = getView(id);
        if (iv != null)
            iv.setBackground(getDrawable(drawable));
    }

    public void setBackground(@IdRes int id, Drawable drawable) {
        View iv = getView(id);
        if (iv != null)
            iv.setBackground(drawable);
    }

    public void setBackgroundColor(@IdRes int id, @ColorInt int color) {
        View iv = getView(id);
        if (iv != null)
            iv.setBackgroundColor(color);
    }

    public void setBackgroundColor(@IdRes int id, @ColorInt int color, float alpha) {
        View iv = getView(id);
        if (iv != null) {
            int a = (int) (alpha * 255);
            iv.setBackgroundColor(a << 24 | (color & 0x00ffffff));
        }
    }

    public void setForeground(@IdRes int id, @DrawableRes int drawable) {
        FrameLayout iv = getView(id);
        if (iv != null)
            iv.setForeground(getDrawable(drawable));
    }

    public void setEnable(@IdRes int id, boolean enable) {
        View iv = getView(id);
        if (iv != null)
            iv.setEnabled(enable);
    }

    public void setActivated(@IdRes int id, boolean enable) {
        View iv = getView(id);
        if (iv != null)
            iv.setActivated(enable);
    }

    public void setFocusable(@IdRes int id, boolean enable) {
        View iv = getView(id);
        if (iv != null)
            iv.setFocusable(enable);
    }

    public void setVisibility(@IdRes int id, int visibility) {
        View view = getView(id);
        if (view != null)
            view.setVisibility(visibility);
    }

    public void setVisibility(@IdRes int id, boolean visible) {
        View view = getView(id);
        if (view != null)
            view.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    public boolean isVisible(@IdRes int id) {
        View view = getView(id);
        return view.getVisibility() == View.VISIBLE;
    }

    public void setElevation(@IdRes int id, float elevation) {
        View view = getView(id);
        if (view != null)
            ViewCompat.setElevation(view, elevation);
    }

    public void setAlpha(int id, @FloatRange(from = 0, to = 1.0) float alpha) {
        View view = getView(id);
        if (view != null)
            view.setAlpha(alpha);
    }

    public void setTranslationX(@IdRes int id, float value) {
        View view = getView(id);
        if (view != null)
            view.setTranslationX(value);
    }

    public void setTranslationY(@IdRes int id, float value) {
        View view = getView(id);
        if (view != null)
            view.setTranslationY(value);
    }

    public float getTranslationY(@IdRes int id) {
        View view = getView(id);
        return view.getTranslationY();
    }

    public void setClickListener(@IdRes int id, View.OnClickListener listener) {
        View view = getView(id);
        if (view != null)
            view.setOnClickListener(listener);
    }

    public void setTopPadding(@IdRes int id, int padding) {
        View view = getView(id);
        if (view != null)
            view.setPadding(view.getPaddingLeft(), padding, view.getPaddingRight(), view.getPaddingBottom());
    }

    public void setHorizontalPadding(@IdRes int id, int padding) {
        View view = getView(id);
        if (view != null)
            view.setPadding(padding, view.getPaddingTop(), padding, view.getPaddingBottom());
    }

    public void setTransitionName(@IdRes int id, String name) {
        View view = getView(id);
        ViewCompat.setTransitionName(view, name);
    }

    public void startAnimation(@IdRes int id, Animation animation) {
        View view = getView(id);
        if (view != null)
            view.startAnimation(animation);
    }

    public void startAnimation(@IdRes int id, @AnimRes int animation) {
        Animation anim = AnimationUtils.loadAnimation(mRootView.getContext(), animation);
        startAnimation(id, anim);
    }

    public void clearAnimation(@IdRes int id) {
        View view = getView(id);
        if (view != null)
            view.clearAnimation();
    }

    public void setMargin(@IdRes int id, int margin, @MarginPosition int position) {
        View view = getView(id);
        ViewGroup.MarginLayoutParams layoutParams = (ViewGroup.MarginLayoutParams) view.getLayoutParams();
        if (haveBits(position, MARGIN_LEFT)) {
            layoutParams.leftMargin = margin;
        }

        if (haveBits(position, MARGIN_RIGHT)) {
            layoutParams.rightMargin = margin;
        }

        if (haveBits(position, MARGIN_TOP)) {
            layoutParams.topMargin = margin;
        }

        if (haveBits(position, MARGIN_BOTTOM)) {
            layoutParams.bottomMargin = margin;
        }

        view.setLayoutParams(layoutParams);
    }

    public int getMargin(@IdRes int id, @MarginPosition int position) {
        View view = getView(id);
        ViewGroup.MarginLayoutParams layoutParams = (ViewGroup.MarginLayoutParams) view.getLayoutParams();
        switch (position) {
            case MARGIN_LEFT:
                return layoutParams.leftMargin;
            case MARGIN_RIGHT:
                return layoutParams.rightMargin;
            case MARGIN_TOP:
                return layoutParams.topMargin;
            case MARGIN_BOTTOM:
                return layoutParams.bottomMargin;
        }
        return 0;
    }

    private boolean haveBits(int val, int cons) {
        return (val & cons) == cons;
    }

    public void addSharedElement(FragmentTransaction transaction, @IdRes int id) {
        View view = getView(id);
        transaction.addSharedElement(view, ViewCompat.getTransitionName(view));
    }

    //一般情况下用这个函数不会有问题
    //如果涉及到typeVar中包含多个类型就会有问题，这种情况不要用这个函数
    public boolean changePasswordVisibility(@IdRes int id) {
        EditText editText = getView(id);
        int selection = editText.getSelectionStart();
        int inputType = editText.getInputType();
        int typeClass = inputType & InputType.TYPE_MASK_CLASS;
        int typeFlag = inputType & InputType.TYPE_MASK_FLAGS;
        int typeVar = inputType & InputType.TYPE_MASK_VARIATION;
        int newType = -1;
        boolean visible = true;
        if (typeClass == InputType.TYPE_CLASS_TEXT) {
            if ((typeVar & InputType.TYPE_TEXT_VARIATION_PASSWORD) == InputType.TYPE_TEXT_VARIATION_PASSWORD) {
                newType = typeClass | typeFlag;
            } else {
                newType = typeClass | typeFlag | InputType.TYPE_TEXT_VARIATION_PASSWORD;
                visible = false;
            }
        } else if (typeClass == InputType.TYPE_CLASS_NUMBER) {
            if ((typeVar & InputType.TYPE_NUMBER_VARIATION_PASSWORD) == InputType.TYPE_NUMBER_VARIATION_PASSWORD) {
                newType = typeClass | typeFlag;
            } else {
                newType = typeClass | typeFlag | InputType.TYPE_NUMBER_VARIATION_PASSWORD;
                visible = false;
            }
        }
        if (newType != -1)
            editText.setInputType(newType);
        editText.setSelection(selection);
        return visible;
    }

    public View getRootView() {
        return mRootView;
    }

    public void addView(@IdRes int id, View child, ViewGroup.LayoutParams layoutParams) {
        ViewGroup viewGroup = getView(id);
        if (layoutParams != null)
            viewGroup.addView(child, layoutParams);
        else
            viewGroup.addView(child);
    }

    public void addView(@IdRes int id, View child) {
        addView(id, child, null);
    }

    public <T extends View> T getView(int id) {
        View textView = mViewArray.get(id);
        if (textView == null) {
            textView = mRootView.findViewById(id);
            mViewArray.put(id, textView);
        }
        //noinspection unchecked
        return (T) textView;
    }

    public int getTop(@IdRes int id) {
        View view = getView(id);
        return view.getTop();
    }

    public int getBottom(@IdRes int id) {
        View view = getView(id);
        return view.getBottom();
    }

    public int getLeft(@IdRes int id) {
        View view = getView(id);
        return view.getLeft();
    }

    public int getRight(@IdRes int id) {
        View view = getView(id);
        return view.getRight();
    }

    public ViewGroup.LayoutParams getLayoutParams(@IdRes int id) {
        View view = getView(id);
        return view.getLayoutParams();
    }

    public void setLayoutParams(@IdRes int id, ViewGroup.LayoutParams layoutParams) {
        View view = getView(id);
        view.setLayoutParams(layoutParams);
    }

    private Context getContext() {
        return mRootView.getContext();
    }

    private Drawable getDrawable(int res) {
        return res == 0 ? null : AppCompatResources.getDrawable(getContext(), res);
    }

    public void detachFromParent() {
        ViewGroup group = (ViewGroup) mRootView.getParent();
        group.removeView(mRootView);
    }

    public void clear() {
        mViewArray.clear();
    }
}
