package ellax.base.helper;

import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import androidx.annotation.ColorInt;
import androidx.annotation.IntDef;
import android.view.View;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * @author dongdaqing 2018-05-24 上午9:07
 * <p>
 * 如果只是单个GradientDrawable，使用这个工具类更好（相对于xml），
 * 如果包含多种状态（state-list,selector），建议使用xml，更加清晰
 */
public class ShapeHelper {
    public static final int RECTANGLE = 0;
    public static final int OVAL = 1;
    public static final int LINE = 2;
    public static final int RING = 3;

    public static final int CORNER_ALL = 0;
    public static final int CORNER_LEFT_TOP = 1;
    public static final int CORNER_RIGHT_TOP = 2;
    public static final int CORNER_RIGHT_BOTTOM = 4;
    public static final int CORNER_LEFT_BOTTOM = 8;

    public static final int CORNER_TOP = CORNER_LEFT_TOP | CORNER_RIGHT_TOP;
    public static final int CORNER_BOTTOM = CORNER_LEFT_BOTTOM | CORNER_RIGHT_BOTTOM;
    public static final int CORNER_LEFT = CORNER_LEFT_TOP | CORNER_LEFT_BOTTOM;
    public static final int CORNER_RIGHT = CORNER_RIGHT_TOP | CORNER_RIGHT_BOTTOM;

    @IntDef({RECTANGLE, OVAL, LINE, RING})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Shape {
    }

    @IntDef({CORNER_ALL,
            CORNER_LEFT_TOP,
            CORNER_RIGHT_TOP,
            CORNER_RIGHT_BOTTOM,
            CORNER_LEFT_BOTTOM,
            CORNER_TOP,
            CORNER_BOTTOM, CORNER_LEFT, CORNER_RIGHT})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Corner {

    }

    private int shape;
    private float radius;
    private int corner;
    private int solid;
    private int stroke;
    private int strokeColor;
    private int dashWidth;
    private int dashGap;
    private int[] gradients;
    private int gradientType = GradientDrawable.LINEAR_GRADIENT;
    private int width = -1;
    private int height = -1;
    private GradientDrawable.Orientation mOrientation;
    private GradientDrawable mDrawable;

    private ShapeHelper(int shape) {
        this.shape = shape;
    }

    public static ShapeHelper rectangle() {
        return new ShapeHelper(RECTANGLE);
    }

    public static ShapeHelper oval() {
        return new ShapeHelper(OVAL);
    }

    public static ShapeHelper ring() {
        return new ShapeHelper(RING);
    }

    public static ShapeHelper with(GradientDrawable drawable) {
        ShapeHelper helper = new ShapeHelper(RECTANGLE);
        helper.mDrawable = drawable;
        return helper;
    }

    public ShapeHelper shape(@Shape int shape) {
        this.shape = shape;
        return this;
    }

    public ShapeHelper color(@ColorInt int color) {
        this.solid = color;
        return this;
    }

    public ShapeHelper color(String color) {
        this.solid = Color.parseColor(color);
        return this;
    }

    public ShapeHelper corner(@Corner int corner) {
        this.corner = corner;
        return this;
    }

    //radius of corner
    public ShapeHelper radius(float radius) {
        this.radius = radius;
        return this;
    }

    public ShapeHelper stroke(int width) {
        this.stroke = width;
        return this;
    }

    public ShapeHelper strokeColor(int color) {
        this.strokeColor = color;
        return this;
    }

    public ShapeHelper dashWidth(int width) {
        this.dashWidth = width;
        return this;
    }

    public ShapeHelper dashGap(int gap) {
        this.dashGap = gap;
        return this;
    }

    public ShapeHelper gradient(GradientDrawable.Orientation orientation, int[] gradients) {
        this.gradients = gradients;
        this.mOrientation = orientation;
        return this;
    }

    public ShapeHelper gradient(GradientDrawable.Orientation orientation, String... colors) {
        this.mOrientation = orientation;
        if (colors != null) {
            this.gradients = new int[colors.length];

            for (int i = 0; i < this.gradients.length; i++) {
                this.gradients[i] = Color.parseColor(colors[i]);
            }
        }

        return this;
    }

    public ShapeHelper size(int size) {
        return size(size, size);
    }

    public ShapeHelper size(int width, int height) {
        this.width = width;
        this.height = height;
        return this;
    }

    public Drawable make() {
        GradientDrawable drawable = mDrawable;

        if (drawable == null) {
            if (mOrientation != null) {
                drawable = new GradientDrawable(mOrientation, gradients);
            } else {
                drawable = new GradientDrawable();
            }
        }

        //外部传入的drawable最多能使用一次，使用之后释放引用
        mDrawable = null;

        /**
         * 注意：如果使用外部传入的方式进行构造{@link #with(GradientDrawable)}，如果该drawable使用了gradient，
         * 那么在当前对象上也必须设置一次，不然下面的判断会出错
         * <code>
         *      GradientDrawable drawable = new XXXDrawable(Orientation.TOP_BOTTOM,new int[]{...});
         *      ShapeHelper.with(drawable).gradient(Orientation.TOP_BOTTOM)...into(view);
         *
         *      必须将Orientation参数重新设置一遍
         * </code>
         */
        if (mOrientation != null) {
            drawable.setGradientType(gradientType);
        } else {
            drawable.setColor(solid);
        }

        drawable.setShape(shape);
        drawable.setStroke(stroke, strokeColor, dashWidth, dashGap);
        drawable.setSize(width, height);
        if (corner == CORNER_ALL) {
            drawable.setCornerRadius(radius);
        } else {
            float[] radii = new float[8];
            int[] corners = {CORNER_LEFT_TOP, CORNER_RIGHT_TOP, CORNER_RIGHT_BOTTOM, CORNER_LEFT_BOTTOM};
            for (int i = 0; i < corners.length; i++) {
                if ((corner & corners[i]) == corners[i]) {
                    radii[i * 2] = radius;
                    radii[i * 2 + 1] = radius;
                }
            }
            drawable.setCornerRadii(radii);
        }
        return drawable;
    }

    public void into(View... views) {
        if (views != null) {
            for (View v : views) {
                v.setBackground(make());
            }
        }
    }
}
