package com.ella.util.parameterChecking;


import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public class AnnotationUtil {
    private static final Logger logger = LoggerFactory
            .getLogger(AnnotationUtil.class);

    /**
     * 校验格式是否正确，如果@CheckValue() 没有参数，则校验是否为空
     * 例子：@CheckValue("^[A-Za-z0-9_-]{1,32}$") 校验数字、字母、下划线1到32位 ： @CheckValue()
     * 校验字段是否为空
     *
     * @param obj
     * @return 如果返回null表示：校验通过
     */
    public static String checkValue(Object obj) {
        return parseAnnotation(CheckValue.class, obj, true);
    }

    public static String checkBlank(Object obj) {
        return parseNotBlankAnnotation(NotBlank.class, obj, true);
    }

    public static String checkParam(Object obj) {
        parseDefaultValueAnnotation(DefaultValue.class, obj, true);
        String s = parseContainsValueAnnotation(ContainsValue.class, obj, true);
        if (StringUtils.isNotBlank(s)){
            return s;
        }
        return parseNotBlankAnnotation(NotBlank.class, obj, true);
    }

    /**
     * 校验是否为空
     *
     * @param obj
     * @return 如果返回null表示：校验通过
     */
    public static String checkEmpty(Object obj) {
        return parseAnnotation(NotEmpty.class, obj, true);
    }

    private static String parseAnnotation(Class<? extends Annotation> aClazz,
                                          Object obj, boolean hasParent) {
        StringBuilder sb = new StringBuilder();
        boolean flag = false;

        Class<?> clazz = obj.getClass();
        Field[] fields = clazz.getDeclaredFields();

        Field[] bothField = fields;
        if (hasParent) {
            Class<?> superClazz = clazz.getSuperclass();
            Field[] superFields = superClazz.getDeclaredFields();
            bothField = (Field[]) ArrayUtils.addAll(fields, superFields);
        }

        for (Field field : bothField) {
            Annotation annotation = field.getAnnotation(aClazz);
            if (annotation == null)
                continue;
            field.setAccessible(true);

            try {
                if (annotation instanceof CheckValue) {
                    CheckValue cv = (CheckValue) annotation;
                    String regex = cv.value();
                    if (StringUtils.isEmpty(regex)) {
                        // 输入的正则表达式为空，所以不做校验
                        // continue;
                        // NotEmpty ne = (NotEmpty)annotation;
                        Object oValue = field.get(obj);
                        if (oValue == null) {
                            sb.append("字段" + field.getName() + "不能为null|");
                            flag = true;
                        } else {
                            if (oValue instanceof String) {
                                String value = (String) oValue;
                                if (StringUtils.isBlank(value)) {
                                    sb.append("字段" + field.getName() + "不能为空|");
                                    flag = true;
                                }
                            } else {
                                logger.info("字段" + field.getName()
                                        + "不是字符串，不能判断是否为空");
                            }
                        }
                    } else {
                        Pattern pattern = Pattern.compile(regex);
                        String value = (String) field.get(obj);
                        Matcher m = pattern.matcher(value);
                        if (!m.matches()) {
                            sb.append("字段" + field.getName() + "格式错误|");
                            flag = true;
                        }
                    }

                } else if (annotation instanceof NotEmpty) {
                    Object oValue = field.get(obj);
                    if (oValue == null) {
                        sb.append("字段" + field.getName() + "不能为null|");
                        flag = true;
                    } else {
                        if (oValue instanceof String) {
                            String value = (String) oValue;
                            if (StringUtils.isBlank(value)) {
                                sb.append("字段" + field.getName() + "不能为空|");
                                flag = true;
                            }
                        } else {
                            logger.info("字段" + field.getName()
                                    + "不是字符串，不能判断是否为空");
                        }
                    }

                }
            } catch (Exception e) {
                sb.append(e.getMessage());
                flag = true;
                logger.error("解析注解出错：", e);
                // e.printStackTrace();
            }

        }

        if (flag) {
            return sb.toString();
        } else {
            return null;
        }
    }

    private static String parseNotBlankAnnotation(Class<? extends Annotation> aClazz,
                                                  Object obj, boolean hasParent) {
        StringBuilder sb = new StringBuilder();
        boolean flag = false;

        Class<?> clazz = obj.getClass();
        Field[] fields = clazz.getDeclaredFields();

        Field[] bothField = fields;
        if (hasParent) {
            Class<?> superClazz = clazz.getSuperclass();
            Field[] superFields = superClazz.getDeclaredFields();
            bothField = (Field[]) ArrayUtils.addAll(fields, superFields);
        }

        for (Field field : bothField) {
            Annotation annotation = field.getAnnotation(aClazz);
            if (annotation == null)
                continue;
            field.setAccessible(true);

            try {
                if (annotation instanceof NotBlank) {
                    NotBlank blankAnnotation = (NotBlank) annotation;
                    Object oValue = field.get(obj);
                    if (oValue == null) {
                        appendValue(sb, blankAnnotation);
                        flag = true;
                    } else {
                        if (oValue instanceof String) {
                            String value = (String) oValue;
                            if (StringUtils.isBlank(value)) {
                                appendValue(sb, blankAnnotation);
                                flag = true;
                            }
                        } else {
                            logger.info(blankAnnotation.value());
                        }
                    }
                }
            } catch (Exception e) {
                sb.append(e.getMessage());
                flag = true;
                logger.error("解析注解出错：", e);
                // e.printStackTrace();
            }
        }
        if (StringUtils.isNotBlank(sb.toString())) {
            sb.append("。");
        }

        if (flag) {
            return sb.toString();
        } else {
            return null;
        }
    }

    private static String parseContainsValueAnnotation(Class<? extends Annotation> aClazz,
                                                       Object obj, boolean hasParent) {
        StringBuilder sb = new StringBuilder();
        boolean flag = false;

        Class<?> clazz = obj.getClass();
        Field[] fields = clazz.getDeclaredFields();

        Field[] bothField = fields;
        if (hasParent) {
            Class<?> superClazz = clazz.getSuperclass();
            Field[] superFields = superClazz.getDeclaredFields();
            bothField = (Field[]) ArrayUtils.addAll(fields, superFields);
        }

        for (Field field : bothField) {
            Annotation annotation = field.getAnnotation(aClazz);
            if (annotation == null)
                continue;
            field.setAccessible(true);

            try {
                if (annotation instanceof ContainsValue) {
                    ContainsValue defaultValue = (ContainsValue) annotation;
                    Object oValue = field.get(obj);
                    if (field.getType() == String.class) {
                        String value = (String) oValue;
                        String v = defaultValue.value();
                        Set<String> set = v == null ? new HashSet<>() : new HashSet<>(Arrays.asList(v.split(",")));
                        if (!set.contains(value)) {
                            appendValue(sb, defaultValue);
                        }
                    } else {
                        continue;
                    }
                }
            } catch (Exception e) {
                sb.append(e.getMessage());
                flag = true;
                logger.error("解析注解出错：", e);
                // e.printStackTrace();
            }
        }
        if (StringUtils.isNotBlank(sb.toString())) {
            sb.append("。");
        }

        if (flag) {
            return sb.toString();
        } else {
            return null;
        }
    }


    private static void parseDefaultValueAnnotation(Class<? extends Annotation> aClazz,
                                                    Object obj, boolean hasParent) {

        Class<?> clazz = obj.getClass();
        Field[] fields = clazz.getDeclaredFields();
        Method[] methods = clazz.getDeclaredMethods();

        Field[] bothField = fields;
        Method[] bothMethods = methods;
        if (hasParent) {
            Class<?> superClazz = clazz.getSuperclass();
            Field[] superFields = superClazz.getDeclaredFields();
            bothField = (Field[]) ArrayUtils.addAll(fields, superFields);

            Method[] superMethods = superClazz.getDeclaredMethods();
            bothMethods = ArrayUtils.addAll(methods, superMethods);
        }

        for (Field field : bothField) {
            Annotation annotation = field.getAnnotation(aClazz);
            if (annotation == null)
                continue;
            field.setAccessible(true);

            try {
                if (annotation instanceof DefaultValue) {
                    DefaultValue defaultValue = (DefaultValue) annotation;

                    Object oValue = field.get(obj);
                    if (field.getType() == String.class) {
                        String value = (String) oValue;
                        if (StringUtils.isBlank(value)) {
                            String v = defaultValue.value();
                            if (StringUtils.isNotBlank(v)) {
                                String name = parSetName(field.getName());
                                Method method = clazz.getMethod(name, String.class);
                                method.invoke(obj, v);
                            }
                        }
                    } else if (field.getType() == Integer.class) {
                        Integer value = (Integer) oValue;
                        if (null == value) {
                            int v = defaultValue.num();
                            String name = parSetName(field.getName());
                            Method method = clazz.getMethod(name, Integer.class);
                            method.invoke(obj, v);
                        }
                    } else if (field.getType() == Date.class) {
                        Date value = (Date) oValue;
                        if (null == value) {
                            String name = parSetName(field.getName());
                            Method method = clazz.getMethod(name, Date.class);
                            method.invoke(obj, new Date());
                        }
                    } else {
                        continue;
                    }
                }
            } catch (Exception e) {
                logger.error("解析注解出错：", e);
            }
        }

    }

    public static String parSetName(String fieldName) {
        if (null == fieldName || "".equals(fieldName)) {
            return null;
        }
        int startIndex = 0;
        if (fieldName.charAt(0) == '_')
            startIndex = 1;
        return "set"
                + fieldName.substring(startIndex, startIndex + 1).toUpperCase()
                + fieldName.substring(startIndex + 1);
    }

    private static void appendValue(StringBuilder sb, NotBlank annotation) {
        if (StringUtils.isBlank(sb.toString())) {
            sb.append(annotation.value());
        } else {
            sb.append("；" + annotation.value());
        }
    }

    private static void appendValue(StringBuilder sb, ContainsValue annotation) {
        if (StringUtils.isBlank(sb.toString())) {
            sb.append(annotation.value());
        } else {
            sb.append("；" + annotation.value());
        }
    }

}