package com.ella.util.excelUtil;

import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author: wqq
 * @Date: 2021/4/25 14:29
 * @Description:
 * @Modified:
 */
public class ExcelUtils {

    private static final Logger logger = LoggerFactory.getLogger(ExcelUtils.class);

    private final static String XLS="xls";

    private final static String XLSX="xlsx";

    /**
     * 导入数据
     *
     * @param file
     * @param fileHead    指定导入的列   若不指定, 则默认读取文档的表头所在列的所有数据
     * @return Map
     * code:  int  200 代表成功, 其他代表失败
     * errorMessage:  String  失败信息
     * result:  List<JSONObject>  (key = 1,2,3,4,5,6...)
     */
    public static Map<String, Object> readExcel(final MultipartFile file, String... fileHead) {
        Map<String, Object> resultMap = new HashMap<>(4);
        resultMap.put("code", 300);
        InputStream inputStream;
        try {
            inputStream = file.getInputStream();
        } catch (Exception e) {
            resultMap.put("errorMessage", "InputStream is null");
            return resultMap;
        }
        Workbook wb;
        if (inputStream == null) {
            resultMap.put("errorMessage", "InputStream is null");
            return resultMap;
        }

        // 获取Workbook
        Map<String, Object> wbResult = getWorkbook(file,inputStream);
        if ((int) wbResult.get("code") != 200) {
            return wbResult;
        }
        wb = (Workbook) wbResult.get("result");

        // 校验表头
        if (fileHead != null && fileHead.length != 0) {
            String checkHead = excelCheckHead(wb, fileHead);
            if (StringUtils.isNotBlank(checkHead)) {
                resultMap.put("errorMessage", checkHead);
                return resultMap;
            }
        }
        // 读取数据
        resultMap = excelReadData(wb, fileHead);
        return resultMap;
    }

    /**
     * 创建Workbook
     *
     * @param inputStream
     * @return
     */
    private static Map<String, Object> getWorkbook(MultipartFile file,InputStream inputStream) {
        int len;
        Workbook wb = null;
        byte[] bufferIs = new byte[1024];
        Map<String, Object> resultMap = new HashMap<>(4);
        resultMap.put("code", 301);
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        try {
            while ((len = inputStream.read(bufferIs)) > -1) {
                byteArrayOutputStream.write(bufferIs, 0, len);
            }
            byteArrayOutputStream.flush();
        } catch (Exception e) {
            logger.error("create Workbook error:" + e.getMessage(), e);
            resultMap.put("errorMessage", e.getMessage());
            return resultMap;
        } finally {
            try {
                inputStream.close();
                byteArrayOutputStream.close();
            } catch (Exception e) {
                logger.error("close stream error:" + e.getMessage(), e);
            }
        }
        InputStream temp1 = new ByteArrayInputStream(byteArrayOutputStream.toByteArray());
        // 目前只读取xls, 不读取xlsx
        String originalFilename = file.getOriginalFilename();
        try {
            if(originalFilename.endsWith(XLS)){
                wb = new HSSFWorkbook(temp1);
            }else if(originalFilename.endsWith(XLSX)){
                wb = new XSSFWorkbook(temp1);
            }
        } catch (Exception e) {
            logger.error("create HSSFWorkbook error:" + e.getMessage(), e);
            resultMap.put("code", 401);
            resultMap.put("errorMessage", "文件格式错误，请按模板填写上传");
            try {
                wb = new XSSFWorkbook(temp1);
            } catch (IOException e1) {
                e1.printStackTrace();
                logger.error("create XSSFWorkbook error:" + e1.getMessage(), e1);
            }
            //return resultMap;
        } finally {
            try {
                temp1.close();
            } catch (Exception e) {
                logger.error("close ByteArrayInputStream error:" + e.getMessage(), e);
            }
        }
        if(wb == null){
            return resultMap;
        }
        resultMap.put("code", 200);
        resultMap.put("result", wb);
        return resultMap;
    }

    /**
     * 表头校验  必须包含要求的表头, 可以多余要求的表头
     *
     * @param wb
     * @param sheetHead
     * @return
     */
    private static String excelCheckHead(Workbook wb, String[] sheetHead) {
        // 验证填写说明
        Sheet sheet = wb.getSheetAt(0);
        Row row = sheet.getRow(0);
        if(!row.getCell(0).getStringCellValue().contains("说明")){
            return "文件格式错误，请按模板填写上传";
        }
        row = sheet.getRow(1);
        // 校验表头
        for(int i = 0;i < sheetHead.length;i++){
            if(!row.getCell(i).getStringCellValue().equals(sheetHead[i])){
                return "表头错误，请按模板填写上传";
            }
        }
        int sheetsNum = wb.getNumberOfSheets();
        for (int i = 1; i < sheetsNum; i++) {
            sheet = wb.getSheetAt(i);
            // 如果校验表头, 有数据的工作表的第一行必须是表头
            if (sheet != null && sheet.getPhysicalNumberOfRows() > 1) {
                row = sheet.getRow(0);
                int lastCellNum = row.getLastCellNum();
                while (lastCellNum > 0 && (row.getCell(lastCellNum - 1) == null || "".equals(row.getCell(lastCellNum - 1).toString().trim()))) {
                    lastCellNum--;
                }
                if (lastCellNum < sheetHead.length) {
                    return sheet.getSheetName() + "工作表的表头(第一行数据)与要求表头不一致, 缺少"+sheetHead[lastCellNum]+"列";
                }
                boolean existFlag;
                for (String headStr : sheetHead) {
                    existFlag = false;
                    for (Cell cell : row) {
                        if (headStr.equals(cell.toString().trim())) {
                            existFlag = true;
                            break;
                        }
                    }
                    if (!existFlag) {
                        return sheet.getSheetName() + "工作表的表头(第一行数据)与要求表头不一致, 缺少"+headStr+"列";
                    }
                }
            }
        }
        return null;
    }

    /**
     * 读取 excel 表格的数据
     *
     * @param wb
     * @param sheetHead
     * @return
     */
    private static Map<String, Object> excelReadData(Workbook wb, String[] sheetHead) {
        List<JSONObject> list = new ArrayList<>(16);
        Map<String, Object> resultMap = new HashMap<>(4);
        resultMap.put("code", 302);

        int sheetsNum = wb.getNumberOfSheets();
        for (int i = 0; i < sheetsNum; i++) {
            Sheet sheet = wb.getSheetAt(i);
            if (sheet.getPhysicalNumberOfRows() <= 1) {
                continue;
            }
            // 如果表头不存在, 需要读取当前工作表的表头
            if (sheetHead == null || sheetHead.length == 0) {
                sheetHead = excelSheetHead(sheet);
                if (sheetHead == null || sheetHead.length == 0) {
                    resultMap.put("errorMessage", sheet.getSheetName() + "工作表的表头信息不存在");
                    return resultMap;
                } else {
                    if (sheetHead[0] == null || "".equals(sheetHead[0].trim())) {
                        resultMap.put("errorMessage", sheet.getSheetName() + "工作表的表头存在空列, 请处理");
                        return resultMap;
                    }
                }
            }

            int temp = 2;
            for (int m = 2; m < sheet.getPhysicalNumberOfRows(); m++) {
                JSONObject jsonObject = new JSONObject();
                Row row = sheet.getRow(temp);
                while (row == null) {
                    row = sheet.getRow(++temp);
                }

                int lastCellNum = row.getLastCellNum();

               // 去除空行问题
                while (lastCellNum > 0 && (row.getCell(lastCellNum - 1) == null || "".equals(row.getCell(lastCellNum - 1).toString().trim()))) {
                    lastCellNum--;
                }
                // 空行直接不处理，继续解析下一行
                if (lastCellNum == 0) {
                    ++temp;
                    continue;
                }
                ++temp;

                jsonObject.put("row", temp);
                jsonObject.put("sheet", sheet.getSheetName());
                for (Cell cell : row) {
                    if (cell != null && !"".equals(cell.toString().trim())) {
                        jsonObject.put(cell.getColumnIndex() + "", String.valueOf(cell));
                    }
                }
                list.add(jsonObject);
            }
        }
        if (CollectionUtils.isEmpty(list)) {
            resultMap.put("errorMessage", "excel表格没有数据");
        } else {
            resultMap.put("code", 200);
            resultMap.put("result", list);
        }
        return resultMap;
    }

    private static String[] excelSheetHead(Sheet sheet) {
        if (sheet == null || sheet.getPhysicalNumberOfRows() < 1) {
            return null;
        }
        Row row = sheet.getRow(0);
        if (row == null || row.getPhysicalNumberOfCells() == 0) {
            return null;
        }

        int count = 0;
        for (Cell cell : row) {
            if (cell != null && !"".equals(cell.toString().trim())) {
                count++;
            }
        }

        String[] head = new String[count];
        int lastCellNum = row.getLastCellNum();
        while (lastCellNum > 0 && (row.getCell(lastCellNum - 1) == null || "".equals(row.getCell(lastCellNum - 1).toString().trim()))) {
            lastCellNum--;
        }

        if (lastCellNum == 0) {
            return null;
        }

        if (lastCellNum > head.length) {
            return head;
        }

        int i = 0;
        for (Cell cell : row) {
            if (cell != null && !"".equals(cell.toString().trim())) {
                head[i++] = cell.toString().trim();
            }
        }
        return head;
    }
}
