package com.ella.util;

import java.text.ParseException;
import java.util.Date;
import java.util.Objects;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.FastDateFormat;

import com.ella.errorCode.HomeErrorCode;
import com.ella.exception.EllaBookException;
import com.ella.response.ResponseParams;

public class ExceptionMsgUtil {
	private static final FastDateFormat YYYY_MM_DD = FastDateFormat.getInstance("yyyy-MM-dd");

	private static final String UNLIMIT_START_DATE = "2018-12-31";

	private static final long MILLISECOND_PER_DAY = 24 * 60 * 60 * 1000;

	public static void fillMsg(Exception e, ResponseParams<?> resp) {
		if (e instanceof EllaBookException) {
			resp.fillError(e.getMessage(), HomeErrorCode.REQUEST_ERROR_CODE);
		} else if (e instanceof ParseException) {
			resp.fillError("格式转换错误", HomeErrorCode.REQUEST_ERROR_CODE);
		} else {
			resp.fillError(HomeErrorCode.SYSTEM_ERROR_MSG, HomeErrorCode.SYSTEM_ERROR_CODE);
		}
	}

	public static void requireTrue(boolean check, String msg) {
		if (!check) {
			throw new EllaBookException(msg);
		}
	}

	public static void requireNonNull(Object obj, String msg) {
		requireTrue(Objects.nonNull(obj), msg);
	}

	public static void requireNotBlank(String str, String msg) {
		requireTrue(StringUtils.isNotBlank(str), msg);
	}

	public static void DateLimit(Date startTime, Date endTime, int days) {
		DateLimit(startTime, endTime, days, false);
	}

	/**
	 * 
	 * @param startTime
	 * @param endTime
	 * @param days
	 * @param unlimitByStartTime 是否根据开始时间=2018-12-31排除限制
	 */
	public static void DateLimit(Date startTime, Date endTime, int days, boolean unlimitByStartTime) {
		if (!ObjectUtils.allNotNull(startTime, endTime)) {
			throw new EllaBookException("时间不能为空！");
		}
		if (days < 1) {
			return;
		}

		if (unlimitByStartTime && UNLIMIT_START_DATE.equals(YYYY_MM_DD.format(startTime))) {
			return;
		}
		if (endTime.getTime() - startTime.getTime() > (days * MILLISECOND_PER_DAY)) {
			throw new EllaBookException("查询时间段过长");
		}
	}
	
	public static void requireTrueWithResponse(boolean check, String msg) {
		if (!check) {
			ResponseParams<?> error = new ResponseParams<>();
			error.fillError(msg, "50000000");
			throw new EllaBookException(error);
		}
	}
	
	public static void requireNotBlankWithResponse(String str , String msg) {
		requireTrueWithResponse(StringUtils.isNotBlank(str), msg);
	}
}
