package com.ella.aspect;

import com.ella.response.ResponseParams;
import com.ella.util.ApplicationContextHelper;
import com.ella.util.redisnew.DistributedCache;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;

/**
 * 并发处理的切面
 *
 * @author LiBin
 * @email libin@ellabook.cn
 * @create 2018-01-11 15:04
 **/
public class ConcurrentAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConcurrentAspect.class);

    private ResponseParams responseParamsUtil = new ResponseParams();

    protected static final long expireTime = 10;

    protected DistributedCache redis;

    @Pointcut("@annotation(AvoidConcurrent)")
    public void concurrentPointcut() {
    }

    @Around("concurrentPointcut()")
    public Object Interceptor(ProceedingJoinPoint joinPoint) throws Exception {
        String clazzName = joinPoint.getTarget().getClass().getName();
        String methodName = joinPoint.getSignature().getName();
        Object[] args = joinPoint.getArgs();
        StringBuilder builder = new StringBuilder();
        String redisKey = "";
        redis = getRedis();
        Method method = ((MethodSignature) joinPoint.getSignature()).getMethod();
        AvoidConcurrent avoidConcurrent = method.getAnnotation(AvoidConcurrent.class);
        String[] primaryKey = avoidConcurrent.primaryKey();
        if (primaryKey.length > 0) {
            builder.append(methodName);
            redisKey = avoidConcurrent.keyHelperClz().newInstance().getPrimaryKey(joinPoint);
            builder.append(redisKey);
            redisKey = builder.toString();
        }

        if (primaryKey.length < 1 || StringUtils.isBlank(redisKey)) {
            redisKey = getMethodKey(clazzName, methodName, args);
        }

        LOGGER.info("method {} concurrentPointcut key {}", methodName, redisKey);

        try {
            if (null == redis) {
                return (ResponseParams) joinPoint.proceed(args);
            }
            if (!StringUtils.isBlank(getRedisKey(redisKey, methodName))) {
                LOGGER.info("method {} concurrentPointcut get key {}", methodName, redis.get(redisKey));
                return responseParamsUtil.error("请稍后...", null);
            }
            setRedisKey(redisKey, methodName);
            return joinPoint.proceed(args);
        } catch (Throwable e) {
            LOGGER.error("系统异常", e);
            return responseParamsUtil.error("系统异常", "系统异常");
        } finally {
            deleteRedisKey(redisKey, methodName);
        }
    }

    private DistributedCache getRedis() {
        try {
            return ApplicationContextHelper.getBean(DistributedCache.class);
        } catch (Exception e) {
            return null;
        }
    }

    private String getRedisKey(String redisKey, String methodName) {
        try {
            LOGGER.info("method {} concurrentPointcut get key {}", methodName, redisKey);
            return redis.get(redisKey);
        } catch (Exception e) {
            LOGGER.error("method {} concurrentPointcut get key error{}", methodName, e);
            return "";
        }
    }

    private void setRedisKey(String redisKey, String methodName) {
        try {
            LOGGER.info("method {} concurrentPointcut set key {}", methodName, redisKey);
            redis.setNX(redisKey, methodName);
            redis.expire(redisKey, expireTime);
        } catch (Exception e) {
            LOGGER.error("method {} concurrentPointcut set key error{}", methodName, e);
        }
    }

    private void deleteRedisKey(String redisKey, String methodName) {
        if (null == redis) {
            return;
        }
        try {
            LOGGER.info("method {} concurrentPointcut del key {}", methodName, redisKey);
            redis.del(redisKey);
        } catch (Exception e) {
            LOGGER.error("method {} concurrentPointcut del key error{}", methodName, e);
        }
    }

    public static String getMethodKey(String clazzName, String methodName, Object[] args) {
        StringBuilder key = new StringBuilder(clazzName);
        key.append(methodName);

        for (Object obj : args) {
            key.append(obj.toString().replace(".", "")
                    .replace(" ", "")
                    .replace(",", "")
                    .replace("\"", "")
                    .replace(":", "")
            );
        }
        return key.toString();
    }
}

